// Define the BackendError interface
// interface BackendError {
//   statusCode: number;
//   message: string;
// }

// Define error codes as an enum to match the backend
enum ErrorCode {
  UNAUTHORIZED = 600,
  INVALID_CREDENTIALS = 601,
  VALIDATION_ERROR = 602,
  ERROR_FILES_MUST_BE_SEND = 603,
  ERROR_FILE_NOT_EXIST = 605,
  // ERROR_MUST_ENTER_HEADER_OR_PARENT_INFO = 605,
}

// Define a mapping of status codes to messages
const errorMessages: Record<ErrorCode, Record<string, string> | string> = {
  [ErrorCode.UNAUTHORIZED]: {
    DEFAULT: 'غير مصرح لك بالوصول، يرجى تسجيل الدخول',
  },
  [ErrorCode.INVALID_CREDENTIALS]: {
    INVALID_CREDENTIALS: 'اسم المستخدم أو كلمة المرور غير صحيحة',
    DEFAULT: 'حدث خطأ أثناء تسجيل الدخول',
  },
  [ErrorCode.VALIDATION_ERROR]: {
    VALIDATION_ERROR: 'البيانات المدخلة غير صالحة، يرجى التحقق من الحقول',
    DEFAULT: 'خطأ في التحقق من البيانات',
  },
  [ErrorCode.ERROR_FILES_MUST_BE_SEND]: {
    ERROR_FILES_MUST_BE_SEND: 'يجب إرسال الملفات المطلوبة',
    DEFAULT: 'خطأ في إرسال الملفات',
  },
  [ErrorCode.ERROR_FILE_NOT_EXIST]: {
    ERROR_FILE_NOT_EXIST: 'الملف غير موجود',
    ERROR_MUST_ENTER_HEADER_OR_PARENT_INFO: 'يجب إدخال معلومات العنوان أو الوالدين',
    DEFAULT: 'خطأ غير معروف في التعامل مع الملفات أو المعلومات',
  },
};

// Main error handler function
// eslint-disable-next-line @typescript-eslint/no-explicit-any
export const handleBackendError = (error: any): string => {
  if (error) {
    // Check if error has statusCode and message directly
    if (error.statusCode && error.message) {
      const { statusCode, message } = error;

      // Check if the statusCode exists in the errorMessages mapping
      if (statusCode in errorMessages) {
        const messages = errorMessages[statusCode as ErrorCode];

        // If the messages are an object, look for a specific message match
        if (typeof messages === 'object') {
          return messages[message] || messages.DEFAULT || 'حدث خطأ غير معروف';
        }

        // If the message is a string (no nested messages), return it
        return messages as string;
      }

      // Fallback for unknown status codes
      return 'حدث خطأ غير معروف، يرجى المحاولة لاحقًا';
    }

    // If error.message exists but is not a JSON string, handle it as a plain message
    if (error.message) {
      return error.message || 'حدث خطأ غير معروف';
    }
  }

  return 'حدث خطأ غير متوقع';
};
