import { NextRequest, NextResponse } from 'next/server';
import { jwtVerify, JWTVerifyResult } from 'jose'; // Use 'jose' for JWT verification

// Fetch the login key from the backend
async function fetchLoginKey(): Promise<string | null> {
  try {
    const response = await fetch(`${process.env.NEXT_PUBLIC_SERVER_URL}/user/me/login-key`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
      },
    });
    if (!response.ok) {
      throw new Error('Failed to fetch login key');
    }
    const data = await response.json();
    return data.key || null;
  } catch (error) {
    console.error('Error fetching login key:', error);
    return null;
  }
}

// Verify JWT token
async function verifyToken(token: string): Promise<boolean> {
  try {
    const secret = new TextEncoder().encode(process.env.JWT_SECRET || 'your_jwt_secret'); // Match backend JWT_SECRET
    const { payload }: JWTVerifyResult = await jwtVerify(token, secret, {
      issuer: 'docProfileHub', // Match issuer from backend
      subject: 'iam@user.me',  // Match subject from backend
    });
    const currentTime = Math.floor(Date.now() / 1000); // Current time in seconds
    if (payload.exp && payload.exp < currentTime) {
      return false; // Token is expired
    }
    return true; // Token is valid and not expired
  } catch (error) {
    console.error('Token verification failed:', error);
    return false; // Invalid or malformed token
  }
}

export async function middleware(request: NextRequest) {
  const { pathname, searchParams } = request.nextUrl;

  // Get the auth_token from cookies
  const token = request.cookies.get('dr_auth_token')?.value;

  // Fetch the login key dynamically
  const loginKey = await fetchLoginKey();

  // Logic for /login route
  if (pathname === '/login') {
    if (token) {
      // Verify token before redirecting to dashboard
      const isValidToken = await verifyToken(token);
      if (isValidToken) {
        return NextResponse.redirect(new URL('/dashboard', request.url));
      }
      // If token is invalid or expired, proceed to login page (allow re-login)
    }

    // If not logged in, proceed with the key check
    const key = searchParams.get('key');
    if (loginKey && key === loginKey) {
      return NextResponse.next(); // Allow access to /login if key matches
    } else {
      return NextResponse.redirect(new URL('/', request.url)); // Redirect to homepage if key is invalid
    }
  }

  // Logic for /dashboard route
  if (pathname.startsWith('/dashboard')) {
    if (!token) {
      // No token, redirect to login with key if available
      const redirectUrl = loginKey ? `/login?key=${loginKey}` : '/login';
      return NextResponse.redirect(new URL(redirectUrl, request.url));
    }

    // Verify token for dashboard access
    const isValidToken = await verifyToken(token);
    if (!isValidToken) {
      // Token is invalid or expired, clear cookie and redirect to login
      const response = NextResponse.redirect(new URL(loginKey ? `/login?key=${loginKey}` : '/login', request.url));
      response.cookies.delete('dr_auth_token');
      return response;
    }

    // If token is valid and accessing /dashboard exactly, redirect to /dashboard/user
    if (pathname === '/dashboard') {
      return NextResponse.redirect(new URL('/dashboard/user', request.url));
    }

    // Allow access to other /dashboard/* routes
    return NextResponse.next();
  }

  // Allow all other requests to proceed
  return NextResponse.next();
}

export const config = {
  matcher: ['/login', '/dashboard/:path*'], // Matches /login and all /dashboard/* routes
};