'use client';

import { handleBackendError } from '@/shared/api/errorHandler';
import { zodResolver } from '@hookform/resolvers/zod';
import { useMutation } from '@tanstack/react-query';
import Cookies from 'js-cookie';
import { useRouter } from 'next/navigation';
import { useForm } from 'react-hook-form';
import { FaSpinner } from 'react-icons/fa';
import loginSchema, { LoginFormValues } from './login.schema';
import { login } from './login.service';

const LoginForm = () => {
  const router = useRouter();

  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
  } = useForm<LoginFormValues>({
    resolver: zodResolver(loginSchema),
    defaultValues: {
      email: '', // Changed to email
      password: '',
    },
  });

  const { mutate, isPending, isError, error } = useMutation({
    mutationFn: login,
    onSuccess: (data) => {
      reset();

      const token = data?.token;
      if (token) {
        Cookies.set('dr_auth_token', token, { expires: 7, secure: true, sameSite: 'strict' });
        router.push('/dashboard');
      }
    },
    onError: (error) => {
      console.error('Login error:', error);
    },
  });

  const onSubmit = (data: LoginFormValues) => {
    mutate(data);
  };

  return (
    <div className="flex items-center justify-center min-h-screen text-white bg-gradient-to-br from-neutral-100 to-neutral-200">
      <div className="w-full max-w-sm p-8 bg-white rounded-lg shadow-lg">
        <h2 className="mb-6 text-2xl font-bold text-center text-gray-900">تسجيل الدخول</h2>
        <form onSubmit={handleSubmit(onSubmit)} className="space-y-5" dir="rtl">
          <div>
            <label htmlFor="email" className="block mb-2 text-sm font-medium text-gray-700">
              البريد الإلكتروني
            </label>
            <input
              type="email" // Changed type to email
              id="email"
              {...register('email')} // Changed to email
              className="w-full px-4 py-3 text-sm border rounded-lg transition duration-200 text-gray-900 placeholder-gray-500 bg-white border-gray-300 focus:ring-green-500 text-right"
              placeholder="البريد الإلكتروني"
            />
            {errors.email && <p className="mt-1 text-sm text-red-500">{errors.email.message}</p>}
          </div>

          <div>
            <label htmlFor="password" className="block mb-2 text-sm font-medium text-gray-700">
              كلمة المرور
            </label>
            <input
              type="password"
              id="password"
              {...register('password')}
              className="w-full px-4 py-3 text-sm border rounded-lg transition duration-200 text-gray-900 placeholder-gray-500 bg-white border-gray-300 focus:ring-green-500 text-right"
              placeholder="كلمة المرور"
            />
            {errors.password && <p className="mt-1 text-sm text-red-500">{errors.password.message}</p>}
          </div>

          <button
            type="submit"
            className="w-full px-4 py-3 text-sm font-medium text-white bg-sky-600 rounded-lg transition duration-200 hover:bg-sky-700 focus:outline-none focus:ring-2 focus:ring-sky-500 focus:ring-offset-2 disabled:opacity-50 flex items-center justify-center gap-2"
            disabled={isPending}
          >
            {isPending ? <FaSpinner className="animate-spin h-5 w-5" /> : null}
            {isPending ? 'جارٍ تسجيل الدخول...' : 'تسجيل الدخول'}
          </button>

          {isError && <p className="mt-2 text-center text-sm text-red-500">{handleBackendError(error)}</p>}
        </form>
      </div>
    </div>
  );
};

export default LoginForm;
