import QueryProvider from '@/providers';
import type { Metadata } from 'next';
import { cairo, tajawal } from '../config/font';
import { ApiResponse } from './(private)/dashboard/user/user.schema';
import './globals.css';

// Fetch public user details with ISR
async function fetchUserPublicDetails(): Promise<ApiResponse> {
  const response = await fetch(`${process.env.NEXT_PUBLIC_SERVER_URL}/user/me/public`, {
    next: { revalidate: 30 }, // Revalidate every 1 day (86,400 seconds)
  });

  if (!response.ok) {
    throw new Error('Failed to fetch user public details');
  }
  return response.json();
}

// Generate dynamic metadata
export async function generateMetadata(): Promise<Metadata> {
  try {
    const userData = await fetchUserPublicDetails();
    const user = userData.data;
    const userName = user.user_name || 'د. قايد العنزي';
    const logo = user.logo?.[0]?.url || '/logo.svg';

    return {
      title: {
        default: `${userName}`,
        template: `${userName} | %s`,
      },
      description: `موقع ${userName} لنقل الخبرة الطبية في الإشعة التداخلية، علاج آلام العمود الفقري والمفاصل، وتثقيف المرضى حول الأمراض وأساليب العلاج الحديثة.`,
      icons: {
        icon: logo,
      },
    };
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
  } catch (error) {
    // console.error('Error fetching user data for metadata:', error);
    // Fallback metadata in case of failure
    return {
      title: {
        default: 'د. قايد العنزي',
        template: `د. قايد العنزي | %s`,
      },
      description: 'موقع د. قايد العنزي لنقل الخبرة الطبية في الإشعة التداخلية، علاج آلام العمود الفقري والمفاصل، وتثقيف المرضى حول الأمراض وأساليب العلاج الحديثة.',
      icons: {
        icon: '/logo.svg',
      },
    };
  }
}

export default function RootLayout({
  children,
}: Readonly<{
  children: React.ReactNode;
}>) {
  return (
    <html lang="ar" dir="rtl">
      <body className={`${tajawal.variable} ${cairo.variable} antialiased`}>
        <QueryProvider>{children}</QueryProvider>
      </body>
    </html>
  );
}
