import { Metadata } from 'next';
import Link from 'next/link';
import { FaStar } from 'react-icons/fa';

interface Testimonial {
  id: number;
  name: string;
  text: string;
  rate: number;
  createdAt: string;
}

export const metadata: Metadata = {
  title: 'تجارب المرضى',
};

export default async function Testimonials() {
  const API_URL = process.env.NEXT_PUBLIC_SERVER_URL || 'http://localhost:7000';
  let testimonials: Testimonial[] = [];
  let error: string | null = null;

  try {
    const response = await fetch(`${API_URL}/review/list?is_active=true`, {
      method: 'GET',
      next: { revalidate: 30 }, // Revalidate every 30 seconds
    });
    if (!response.ok) throw new Error('فشل في جلب التجارب');
    testimonials = await response.json();
  } catch (err) {
    console.error('Error fetching testimonial reviews:', err);
    error = err instanceof Error ? err.message : 'حدث خطأ غير متوقع';
  }

  return (
    <>
      <section className="py-12 bg-white border border-neutral-200 rounded-lg">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <h2 className="text-2xl md:text-3xl font-bold font-heading text-doctor-primary mb-6">الآراء والتقييمات العامة</h2>
          <p className="mb-8 text-base text-gray-500 font-body max-w-2xl md:mb-12">اطلع على تجارب المرضى مع خدماتنا الطبية وكيف ساهمت في تحسين حياتهم.</p>
          {error ? (
            <p className="text-center text-red-500 py-6 font-body">{error}. حاول مرة أخرى لاحقًا.</p>
          ) : testimonials.length === 0 ? (
            <p className="text-center text-gray-600 py-6 font-body">لا توجد تجارب متاحة حاليًا.</p>
          ) : (
            <div className="grid gap-6 md:grid-cols-2">
              {testimonials.map((testimonial) => (
                <div key={testimonial.id} className="p-6 rounded-lg bg-white border border-neutral-200 hover:shadow-sm transition-all duration-300 flex flex-col gap-4">
                  <div className="flex gap-1">
                    {[...Array(5)].map((_, index) => (
                      <FaStar key={index} className={index < testimonial.rate ? 'text-yellow-400' : 'text-gray-300'} size={16} />
                    ))}
                  </div>
                  <p className="text-gray-600 text-sm md:text-base font-body">{testimonial.text}</p>
                  <div className="flex justify-between items-center">
                    <p className="text-sm font-semibold text-doctor-dark font-body">{testimonial.name}</p>
                    <p className="text-xs text-gray-500 font-body">{new Date(testimonial.createdAt).toLocaleDateString('ar')}</p>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </section>

      <section className="py-12 bg-doctor-light/20 border border-neutral-200 rounded-lg">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <h2 className="text-3xl md:text-4xl font-bold font-heading text-doctor-dark mb-4">هل ترغب في تجربة رعاية صحية متميزة؟</h2>
          <p className="text-gray-600 text-lg font-body mb-6">تواصل معي اليوم لحجز استشارة أو لمناقشة خيارات العلاج المتاحة لك.</p>
          <Link
            href="/contact"
            className="inline-block bg-doctor-primary text-white px-6 py-3 rounded-lg hover:bg-doctor-dark transition-all duration-300 font-body text-base md:text-lg font-medium"
          >
            تواصل معنا
          </Link>
        </div>
      </section>
    </>
  );
}
