'use client';

import Image from 'next/image';
import Link from 'next/link';
import React, { useState } from 'react';
import { Media, ListMediaFilter, Media_Types } from '@/app/(private)/dashboard/media/media.schema'; // Adjust path

interface MediaVideosClientProps {
  initialMedia: Media[];
  initialTotalItems: number;
  initialFilter: ListMediaFilter;
}

const MediaVideosClient: React.FC<MediaVideosClientProps> = ({ initialMedia, initialTotalItems, initialFilter }) => {
  const [mediaItems, setMediaItems] = useState<Media[]>(initialMedia);
  const [totalItems, setTotalItems] = useState(initialTotalItems);
  const [filterType, setFilterType] = useState<'all' | Media_Types>('all');
  const [page, setPage] = useState(initialFilter.page);
  const pageSize = initialFilter.pageSize;
  const totalPages = Math.ceil(totalItems / pageSize);

  // Fetch media dynamically when filter or page changes
  const fetchMedia = async (filter: ListMediaFilter, type?: string) => {
    const queryParams = new URLSearchParams({
      page: filter.page.toString(),
      pageSize: filter.pageSize.toString(),
      ...(type && { type }),
    });

    const response = await fetch(`${process.env.NEXT_PUBLIC_SERVER_URL}/media/list?${queryParams.toString()}`, {
      headers: { 'Content-Type': 'application/json' },
    });
    if (!response.ok) {
      throw new Error('Failed to fetch media items');
    }
    const result = await response.json();
    setMediaItems(result.data);
    setTotalItems(result.pagination.totalItemsCount);
  };

  const handleFilterChange = async (type: 'all' | Media_Types) => {
    setFilterType(type);
    setPage(1); // Reset to first page on filter change
    try {
      await fetchMedia({ page: 1, pageSize }, type === 'all' ? undefined : type);
    } catch (error) {
      console.error('Error fetching filtered media:', error);
    }
  };

  const handlePageChange = async (newPage: number) => {
    if (newPage >= 1 && newPage <= totalPages) {
      setPage(newPage);
      try {
        await fetchMedia({ page: newPage, pageSize }, filterType === 'all' ? undefined : filterType);
      } catch (error) {
        console.error('Error fetching paginated media:', error);
      }
    }
  };

  const filteredMedia = filterType === 'all' ? mediaItems : mediaItems.filter((item) => item.media_type === filterType);

  return (
    <>
      {/* Filter Buttons */}
      <div className="flex gap-4 mb-8">
        <button
          onClick={() => handleFilterChange('all')}
          className={`cursor-pointer px-4 py-2 rounded-md font-body font-medium transition-colors duration-300 ${
            filterType === 'all' ? 'bg-doctor-primary text-white' : 'bg-gray-200 text-gray-600 hover:bg-gray-300'
          }`}
        >
          الكل
        </button>
        <button
          onClick={() => handleFilterChange(Media_Types.VIDEO)}
          className={`cursor-pointer px-4 py-2 rounded-md font-body font-medium transition-colors duration-300 ${
            filterType === Media_Types.VIDEO ? 'bg-doctor-primary text-white' : 'bg-gray-200 text-gray-600 hover:bg-gray-300'
          }`}
        >
          الفيديوهات
        </button>
        <button
          onClick={() => handleFilterChange(Media_Types.LECTURE)}
          className={`cursor-pointer px-4 py-2 rounded-md font-body font-medium transition-colors duration-300 ${
            filterType === Media_Types.LECTURE ? 'bg-doctor-primary text-white' : 'bg-gray-200 text-gray-600 hover:bg-gray-300'
          }`}
        >
          محاضرات مرئية
        </button>
        <button
          onClick={() => handleFilterChange(Media_Types.REPORT)}
          className={`cursor-pointer px-4 py-2 rounded-md font-body font-medium transition-colors duration-300 ${
            filterType === Media_Types.REPORT ? 'bg-doctor-primary text-white' : 'bg-gray-200 text-gray-600 hover:bg-gray-300'
          }`}
        >
          تقارير
        </button>
      </div>

      {/* Media Items */}
      <div className="space-y-6">
        {filteredMedia.length > 0 ? (
          filteredMedia
            .filter((item) => item.is_active) // Show only active media
            .map((item) => (
              <div
                key={item.id}
                className="flex flex-col md:flex-row items-center gap-6 p-6 bg-white border border-neutral-200 rounded-lg hover:shadow-sm transition-all duration-300"
              >
                {/* Thumbnail */}
                <Link href={item.url || '#'} target="_blank" rel="noopener noreferrer">
                  <div className="relative w-full md:w-40 h-40 rounded-lg overflow-hidden">
                    {item.thumbnail && item.thumbnail.length > 0 ? (
                      <Image src={item.thumbnail[0].url} alt={item.title} fill className="object-cover" />
                    ) : (
                      <div className="w-full h-full bg-gray-200 flex items-center justify-center">
                        <span className="text-gray-500">بدون صورة</span>
                      </div>
                    )}
                    {/* Play Button Overlay */}
                    <div className="absolute inset-0 flex items-center justify-center">
                      <div className="bg-white/80 rounded-full p-3">
                        <svg className="w-8 h-8 text-doctor-primary" fill="currentColor" viewBox="0 0 24 24">
                          <path d="M8 5v14l11-7z" />
                        </svg>
                      </div>
                    </div>
                  </div>
                </Link>

                {/* Text Content */}
                <div className="flex-1">
                  <h3 className="text-xl font-semibold text-doctor-dark font-heading mb-2">{item.title}</h3>
                  <p className="text-gray-600 text-base font-body leading-relaxed mb-4">{item.description || 'بدون وصف'}</p>
                  <span
                    className={`px-3 py-1 text-sm font-semibold rounded-full ${
                      item.media_type === 'video'
                        ? 'bg-red-100 text-red-600'
                        : item.media_type === 'lecture'
                        ? 'bg-sky-100 text-sky-600'
                        : item.media_type === 'report'
                        ? 'bg-amber-100 text-amber-600'
                        : 'bg-gray-100 text-gray-600'
                    }`}
                  >
                    {item.media_type === 'video' ? 'فيديو' : item.media_type === 'lecture' ? 'محاضرة' : item.media_type === 'report' ? 'تقرير' : 'غير محدد'}
                  </span>
                </div>
              </div>
            ))
        ) : (
          <p className="text-center text-gray-600 font-body">لا توجد عناصر متاحة في هذه الفئة.</p>
        )}
      </div>

      {/* Pagination */}
      {totalPages > 1 && (
        <div className="flex justify-center items-center gap-4 mt-6">
          <button
            onClick={() => handlePageChange(page - 1)}
            disabled={page === 1}
            className="px-4 py-2 bg-doctor-dark text-white rounded-lg disabled:opacity-50 hover:bg-doctor-primary transition-colors"
          >
            السابق
          </button>
          <span className="text-gray-700">
            صفحة {page} من {totalPages}
          </span>
          <button
            onClick={() => handlePageChange(page + 1)}
            disabled={page === totalPages}
            className="px-4 py-2 bg-doctor-dark text-white rounded-lg disabled:opacity-50 hover:bg-doctor-primary transition-colors"
          >
            التالي
          </button>
        </div>
      )}
    </>
  );
};

export default MediaVideosClient;
