import { Account } from '@/app/(private)/dashboard/accounts/accounts.schema'; // Adjust path
import { Workplace } from '@/app/(private)/dashboard/workplaces/workplace.schema'; // Adjust path
import { Metadata } from 'next';
import Link from 'next/link';
import {
  FaEnvelope,
  FaFacebook,
  FaFacebookF,
  FaInstagram,
  FaLink,
  FaLinkedin,
  FaMapMarkerAlt,
  FaMoneyBillWave,
  FaPhoneAlt,
  FaPinterest,
  FaReddit,
  FaSnapchat,
  FaTiktok,
  FaTwitter,
  FaWhatsapp,
  FaYoutube,
} from 'react-icons/fa';

// Fetch workplace items with ISR
async function fetchWorkplaceItems(): Promise<Workplace[]> {
  const response = await fetch(`${process.env.NEXT_PUBLIC_SERVER_URL}/workplace/list`, {
    next: { revalidate: 30 }, // Revalidate every 30 seconds
  });
  if (!response.ok) {
    throw new Error('Failed to fetch workplace items');
  }
  return response.json();
}

// Fetch social media accounts with ISR
async function fetchSocialMediaItems(): Promise<Account[]> {
  const response = await fetch(`${process.env.NEXT_PUBLIC_SERVER_URL}/account/list`, {
    next: { revalidate: 30 }, // Revalidate every 30 seconds
  });
  if (!response.ok) {
    throw new Error('Failed to fetch social media items');
  }
  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  const data = result.data;
  return data as Account[];
}

// Map icon strings to components
const iconMap: { [key: string]: React.ComponentType<{ size?: number }> } = {
  FaFacebookF,
  FaTwitter,
  FaInstagram,
  FaLinkedin,
  FaTiktok,
  FaSnapchat,
  FaFacebook,
  FaYoutube,
  FaWhatsapp,
  FaReddit,
  FaPinterest,
};

export const metadata: Metadata = {
  title: 'تواصل معنا',
};

const Contact = async () => {
  try {
    const [workplaceItems, socialMediaItems] = await Promise.all([fetchWorkplaceItems(), fetchSocialMediaItems()]);

    return (
      <section className="py-12 bg-white border border-neutral-200 rounded-lg min-h-screen" dir="rtl">
        <div className="mx-auto w-full max-w-7xl px-4 sm:px-6 lg:px-8">
          {/* Title and Description */}
          <h2 className="text-2xl md:text-3xl font-bold font-heading text-doctor-primary mb-6">تواصل معنا</h2>
          <p className="mb-8 text-base text-gray-500 font-body max-w-2xl md:mb-12">تواصلوا معنا للحصول على أفضل الخدمات الطبية في أفضل المستشفيات والمراكز الطبية.</p>

          {/* Contact Info Grid */}
          <div className="grid gap-6 sm:grid-cols-2 lg:grid-cols-3">
            {workplaceItems.length > 0 ? (
              workplaceItems.map((info) => (
                <div key={info.id} className="bg-white rounded-lg p-6 flex flex-col gap-4 border border-gray-200 hover:shadow-md transition-all duration-300">
                  <h3 className="text-lg font-semibold text-gray-800 font-heading">{info.name}</h3>
                  <div className="flex items-start gap-3">
                    <FaMapMarkerAlt className="text-cyan-500 mt-1" size={20} />
                    <p className="text-sm text-gray-600 font-body flex-1">{info.address}</p>
                  </div>
                  <div className="flex items-center gap-3">
                    <FaPhoneAlt className="text-cyan-500" size={20} />
                    <p className="text-sm text-gray-600 font-body">{info.phone_1}</p>
                  </div>
                  {info.phone_2 && (
                    <div className="flex items-center gap-3">
                      <FaPhoneAlt className="text-cyan-500" size={20} />
                      <p className="text-sm text-gray-600 font-body">{info.phone_2}</p>
                    </div>
                  )}
                  <div className="flex items-center gap-3">
                    <FaEnvelope className="text-cyan-500" size={20} />
                    <a href={`mailto:${info.email}`} target="_blank" rel="noopener noreferrer" className="text-sm text-cyan-500 hover:underline font-body">
                      {info.email}
                    </a>
                  </div>
                  {info.url && (
                    <div className="flex items-center gap-3">
                      <FaLink className="text-cyan-500" size={20} />
                      <a href={info.url} target="_blank" rel="noopener noreferrer" className="text-sm text-cyan-500 hover:underline font-body">
                        {info.url}
                      </a>
                    </div>
                  )}
                  {info.examine_price && (
                    <div className="flex items-center gap-3">
                      <FaMoneyBillWave className="text-cyan-500" size={20} />
                      <p className="text-sm text-gray-600 font-body">سعر الكشف: {info.examine_price}</p>
                    </div>
                  )}
                </div>
              ))
            ) : (
              <p className="text-center text-gray-600 col-span-full">لا توجد أماكن عمل متاحة حاليًا</p>
            )}
          </div>
        </div>

        {/* Social Media Section */}
        <div className="mt-12 text-center animate-fadeIn" style={{ animationDelay: '0.4s' }}>
          <h3 className="text-xl md:text-2xl font-semibold text-doctor-dark font-heading mb-6">تابعنا على وسائل التواصل الاجتماعي</h3>
          <div className="flex justify-center gap-4">
            {socialMediaItems.filter((social) => social.is_active).length > 0 ? (
              socialMediaItems
                .filter((social) => social.is_active)
                .map((social) => {
                  const IconComponent = iconMap[social.icon];
                  return IconComponent ? (
                    <Link
                      key={social.id}
                      href={social.url}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="flex items-center justify-center w-12 h-12 rounded-full bg-doctor-light/20 text-doctor-primary hover:bg-doctor-primary hover:text-white transition-all duration-300"
                      aria-label={social.icon.replace('Fa', '')}
                    >
                      <IconComponent size={20} />
                    </Link>
                  ) : null;
                })
            ) : (
              <p className="text-gray-600">لا توجد حسابات وسائل تواصل اجتماعي متاحة حاليًا</p>
            )}
          </div>
        </div>
      </section>
    );
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
  } catch (error) {
    return (
      <section className="py-12 bg-white border border-neutral-200 rounded-lg min-h-screen" dir="rtl">
        <div className="mx-auto w-full max-w-7xl px-4 sm:px-6 lg:px-8">
          <h2 className="text-2xl md:text-3xl font-bold font-heading text-doctor-primary mb-6">تواصل معنا</h2>
          <p className="text-center text-red-500">حدث خطأ أثناء جلب البيانات. حاول مرة أخرى لاحقًا.</p>
        </div>
      </section>
    );
  }
};

export default Contact;
