import React from 'react';
import Image from 'next/image';
import { FaMapMarkerAlt, FaPhoneAlt, FaEnvelope, FaMoneyBillWave, FaLink, FaImage } from 'react-icons/fa';
import { ApiResponse } from '@/app/(private)/dashboard/user/user.schema'; // Adjust path
import { Workplace } from '@/app/(private)/dashboard/workplaces/workplace.schema'; // Adjust path
import { Metadata } from 'next';

// Fetch public user details with ISR
async function fetchUserPublicDetails(): Promise<ApiResponse> {
  const response = await fetch(`${process.env.NEXT_PUBLIC_SERVER_URL}/user/me/public`, {
    next: { revalidate: 30 }, // Revalidate every 30 seconds
  });
  if (!response.ok) {
    throw new Error('Failed to fetch user public details');
  }
  return response.json();
}

// Fetch workplace items with ISR
async function fetchWorkplaceItems(): Promise<Workplace[]> {
  const response = await fetch(`${process.env.NEXT_PUBLIC_SERVER_URL}/workplace/list`, {
    next: { revalidate: 30 }, // Revalidate every 30 seconds
  });
  if (!response.ok) {
    throw new Error('Failed to fetch workplace items');
  }
  return response.json();
}

export const metadata: Metadata = {
  title: 'من نحن',
};

export default async function About() {
  try {
    const userData = await fetchUserPublicDetails();
    const workplaceItems = await fetchWorkplaceItems();
    const user = userData.data;

    return (
      <>
        {/* About Us Section */}
        <section className="py-12 bg-white border border-neutral-200 rounded-lg">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
              {/* Text Content */}
              <div className="space-y-6">
                <h2 className="text-2xl md:text-3xl font-bold font-heading text-doctor-primary mb-6">نبذة عنــــــــــــا</h2>
                <p className="text-gray-600 text-sm md:text-base font-body leading-relaxed">
                  {user.about_dr || 'لا توجد معلومات متاحة عن الطبيب في الوقت الحالي. يرجى التحقق لاحقًا للحصول على تفاصيل دقيقة.'}
                </p>
              </div>

              {/* Image */}
              <div className="relative w-full h-64 md:h-80 lg:h-96">
                {user.doctor_pic?.[0]?.url ? (
                  <Image src={user.doctor_pic?.[0]?.url} alt={user.user_name || 'Doctor in clinic'} fill className="object-contain" priority={true} />
                ) : (
                  <div className="w-full h-full bg-gray-100 rounded-lg flex items-center justify-center text-gray-500 my-auto">
                    <FaImage className="text-9xl" />
                  </div>
                )}
              </div>
            </div>
          </div>
        </section>

        {/* Contact Us Section */}
        <section className="py-12 bg-white border border-neutral-200 rounded-lg">
          <div className="mx-auto w-full max-w-7xl px-4 sm:px-6 lg:px-8">
            <h2 className="text-2xl md:text-3xl font-bold font-heading text-doctor-primary mb-6">تواصل معنا</h2>
            <p className="mb-8 text-base text-gray-500 font-body max-w-2xl md:mb-12">تواصلوا معنا للحصول على أفضل الخدمات الطبية في أفضل المستشفيات والمراكز الطبية.</p>
            <div className="grid gap-6 sm:grid-cols-2 lg:grid-cols-3">
              {workplaceItems.length > 0 ? (
                workplaceItems.map((info) => (
                  <div key={info.id} className="bg-white rounded-lg p-6 flex flex-col gap-4 border border-gray-200 hover:shadow-md transition-all duration-300">
                    <h3 className="text-lg font-semibold text-gray-800 font-heading">{info.name}</h3>
                    <div className="flex items-start gap-3">
                      <div className="flex-shrink-0">
                        <FaMapMarkerAlt className="text-cyan-500 mt-1" size={20} />
                      </div>
                      <p className="text-sm text-gray-600 font-body flex-1">{info.address}</p>
                    </div>
                    <div className="flex items-center gap-3">
                      <div className="flex-shrink-0">
                        <FaPhoneAlt className="text-cyan-500" size={20} />
                      </div>
                      <p className="text-sm text-gray-600 font-body">{info.phone_1}</p>
                    </div>
                    {info.phone_2 && (
                      <div className="flex items-center gap-3">
                        <div className="flex-shrink-0">
                          <FaPhoneAlt className="text-cyan-500" size={20} />
                        </div>
                        <p className="text-sm text-gray-600 font-body">{info.phone_2}</p>
                      </div>
                    )}
                    <div className="flex items-center gap-3">
                      <div className="flex-shrink-0">
                        <FaEnvelope className="text-cyan-500" size={20} />
                      </div>
                      <a href={`mailto:${info.email}`} target="_blank" rel="noopener noreferrer" className="text-sm text-cyan-500 hover:underline font-body">
                        {info.email}
                      </a>
                    </div>
                    {info.url && (
                      <div className="flex items-center gap-3">
                        <div className="flex-shrink-0">
                          <FaLink className="text-cyan-500" size={20} />
                        </div>
                        <a href={info.url} target="_blank" rel="noopener noreferrer" className="text-sm text-cyan-500 hover:underline font-body">
                          {info.url}
                        </a>
                      </div>
                    )}
                    {info.examine_price && (
                      <div className="flex items-center gap-3">
                        <div className="flex-shrink-0">
                          <FaMoneyBillWave className="text-cyan-500" size={20} />
                        </div>
                        <p className="text-sm text-gray-600 font-body">سعر الكشف: {info.examine_price}</p>
                      </div>
                    )}
                  </div>
                ))
              ) : (
                <p className="text-center text-gray-600 col-span-full">لا توجد أماكن عمل متاحة حاليًا</p>
              )}
            </div>
          </div>
        </section>
      </>
    );
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
  } catch (error) {
    return (
      <>
        {/* About Us Section (Error Fallback) */}
        <section className="py-12 bg-white border border-neutral-200 rounded-lg">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
              <div className="space-y-6">
                <h2 className="text-xl md:text-2xl font-semibold text-doctor-secondary font-heading">نبذة عنــــــــــــا</h2>
                <p className="text-red-500">حدث خطأ أثناء جلب البيانات. حاول مرة أخرى لاحقًا.</p>
              </div>
              <div className="relative w-full h-64 md:h-80 lg:h-96">
                <Image src="/doctor.webp" alt="Doctor in clinic" fill className="object-contain" priority={true} />
              </div>
            </div>
          </div>
        </section>

        {/* Contact Us Section (Error Fallback) */}
        <section className="py-12 bg-white border border-neutral-200 rounded-lg">
          <div className="mx-auto w-full max-w-7xl px-4 sm:px-6 lg:px-8">
            <h2 className="text-2xl md:text-3xl font-bold font-heading text-doctor-primary mb-6">تواصل معنا</h2>
            <p className="text-center text-red-500">حدث خطأ أثناء جلب البيانات. حاول مرة أخرى لاحقًا.</p>
          </div>
        </section>
      </>
    );
  }
}
