import { FaStar } from 'react-icons/fa';

interface Testimonial {
  id: number;
  name: string;
  text: string;
  rate: number;
  createdAt: string;
}

export default async function TestimonialsSection() {
  const API_URL = process.env.NEXT_PUBLIC_SERVER_URL || 'http://localhost:7000';
  let testimonials: Testimonial[] = [];
  let error: string | null = null;

  try {
    const response = await fetch(`${API_URL}/review/list?limit=10&is_active=true`, {
      method: 'GET',
      next: { revalidate: 30 }, // Revalidate every 30 seconds
    });
    if (!response.ok) throw new Error('فشل في جلب التجارب العامة');
    testimonials = await response.json();
  } catch (err) {
    console.error('Error fetching global reviews:', err);
    error = err instanceof Error ? err.message : 'حدث خطأ غير متوقع';
  }

  return (
    <section className="py-12 bg-white border border-neutral-200 rounded-lg">
      <div className="mx-auto w-full max-w-7xl px-4 sm:px-6 lg:px-8">
        <h2 className="text-3xl md:text-4xl font-bold font-heading text-doctor-primary mb-6">الآراء والتقييمات العامة</h2>
        <p className="mb-8 text-base text-gray-500 font-body max-w-3xl md:mb-12">
          تعرّف على ما يقوله الأطباء والمتخصصون عن خبرة الدكتور ومهارته في علاج آلام العمود الفقري والمفاصل.
        </p>

        {error ? (
          <p className="text-center text-red-500 py-6 font-body">{error}. حاول مرة أخرى لاحقًا.</p>
        ) : testimonials.length === 0 ? (
          <p className="text-center text-gray-600 py-6 font-body">لا توجد آراء متاحة حاليًا.</p>
        ) : (
          <div className="grid gap-6 md:grid-cols-2">
            {testimonials.map((testimonial) => (
              <div key={testimonial.id} className="bg-white rounded-lg p-6 flex flex-col gap-4 border border-gray-200 hover:shadow-sm transition-all duration-300">
                <div className="flex gap-1">
                  {[...Array(5)].map((_, index) => (
                    <FaStar key={index} className={index < testimonial.rate ? 'text-yellow-400' : 'text-gray-300'} size={16} />
                  ))}
                </div>
                <p className="text-sm text-gray-600 font-body">{testimonial.text}</p>
                <div className="flex justify-between items-center">
                  <p className="text-sm font-semibold text-gray-800 font-body">{testimonial.name}</p>
                  <p className="text-xs text-gray-500 font-body">{new Date(testimonial.createdAt).toLocaleDateString('ar')}</p>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>
    </section>
  );
}
