/* eslint-disable @typescript-eslint/no-explicit-any */
/* eslint-disable @typescript-eslint/no-unused-vars */
'use client';

import { FaStar } from 'react-icons/fa';
import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { useMutation } from '@tanstack/react-query';
import { createReviewFormSchema, CreateReviewFormValues } from '@/app/(private)/dashboard/reviews/reviews.schema';
import { createPublicReview } from '@/app/(private)/dashboard/reviews/reviews.service';

export default function ReviewForm() {
  const [successMessage, setSuccessMessage] = useState<string | null>(null);
  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
    setValue,
    watch,
  } = useForm<CreateReviewFormValues>({
    resolver: zodResolver(createReviewFormSchema),
    defaultValues: {
      name: '',
      text: '',
      rate: 1,
      post_id: null, // Set post_id to null for global reviews
      is_active: false, // Static default value for public global reviews
    },
  });

  const rate = watch('rate'); // Watch the rate field for star rating

  const mutation = useMutation({
    mutationFn: createPublicReview,
    onSuccess: () => {
      setSuccessMessage('تم إضافة تقييمك بنجاح!');
      reset({
        name: '',
        text: '',
        rate: 1,
        post_id: null,
        is_active: false, // Reset to false
      });
      setTimeout(() => setSuccessMessage(null), 5000); // Clear message after 5 seconds
    },
    onError: (error: any) => {
      setSuccessMessage(null);
    },
  });

  const onSubmit = (data: CreateReviewFormValues) => {
    mutation.mutate(data);
  };

  const handleStarClick = (rate: number) => {
    setValue('rate', rate);
  };

  return (
    <div className="bg-white border border-neutral-200 rounded-lg p-6">
      <h3 className="text-2xl font-bold text-doctor-dark mb-6 font-heading">أضف تقييمك</h3>
      <form onSubmit={handleSubmit(onSubmit)} className="space-y-4">
        {successMessage && <p className="text-green-600 bg-green-50 p-3 rounded-md text-sm">{successMessage}</p>}
        {mutation.isError && <p className="text-red-600 bg-red-50 p-3 rounded-md text-sm">{mutation.error.message || 'حدث خطأ أثناء إضافة التقييم'}</p>}
        <div>
          <label className="block mb-1 text-sm font-medium text-gray-700">الاسم</label>
          <input
            {...register('name')}
            className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-primary transition-all duration-200"
            placeholder="أدخل اسمك"
          />
          {errors.name && <p className="text-red-500 text-xs mt-1">{errors.name.message}</p>}
        </div>
        <div>
          <label className="block mb-1 text-sm font-medium text-gray-700">التقييم</label>
          <div className="flex gap-1">
            {[1, 2, 3, 4, 5].map((star) => (
              <FaStar key={star} className={`cursor-pointer ${star <= rate ? 'text-yellow-400' : 'text-gray-300'}`} size={24} onClick={() => handleStarClick(star)} />
            ))}
          </div>
          <input type="hidden" {...register('rate')} />
          {errors.rate && <p className="text-red-500 text-xs mt-1">{errors.rate.message}</p>}
        </div>
        <div>
          <label className="block mb-1 text-sm font-medium text-gray-700">التعليق</label>
          <textarea
            {...register('text')}
            className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-primary transition-all duration-200 min-h-[100px]"
            placeholder="أدخل تعليقك"
          />
          {errors.text && <p className="text-red-500 text-xs mt-1">{errors.text.message}</p>}
        </div>
        <button
          type="submit"
          disabled={mutation.isPending}
          className="w-full px-4 py-2 bg-doctor-primary text-white rounded-lg hover:bg-doctor-dark disabled:opacity-50 transition-colors duration-200"
        >
          {mutation.isPending ? 'جارٍ الإرسال...' : 'إرسال التقييم'}
        </button>
      </form>
    </div>
  );
}
