'use client';

import Link from 'next/link';
import { useState } from 'react';
import { FaChevronDown, FaChevronUp } from 'react-icons/fa';
import { MdKeyboardArrowLeft } from 'react-icons/md';

export interface Post {
  id: number;
  title: string;
  link: string;
}

export interface Subcategory {
  id: number;
  title: string;
  posts: Post[];
}

export interface Department {
  id: number;
  title: string;
  subcategories: Subcategory[];
}

interface DepartmentsAsideClientProps {
  departments: Department[];
}

const DepartmentsAsideClient = ({ departments }: DepartmentsAsideClientProps) => {
  const [openDepartmentIndex, setOpenDepartmentIndex] = useState<number | null>(null);
  const [openSubcategoryIndex, setOpenSubcategoryIndex] = useState<number | null>(null);

  const toggleDepartment = (index: number) => {
    setOpenDepartmentIndex(openDepartmentIndex === index ? null : index);
    setOpenSubcategoryIndex(null); // Reset subcategory when toggling department
  };

  const toggleSubcategory = (index: number) => {
    setOpenSubcategoryIndex(openSubcategoryIndex === index ? null : index);
  };

  return (
    <ul className="space-y-3">
      {departments.map((department, deptIndex) => (
        <li key={department.id} className="bg-gray-50 rounded-lg overflow-hidden">
          {/* Department Toggle */}
          <button
            onClick={() => toggleDepartment(deptIndex)}
            className="w-full flex justify-between items-center py-3 px-4 text-gray-800 hover:bg-gray-100 focus:bg-gray-100 transition-colors duration-200 focus:outline-none focus:ring-2 focus:ring-doctor-primary"
          >
            <span className="text-sm font-semibold font-body">{department.title}</span>
            {openDepartmentIndex === deptIndex ? <FaChevronUp className="text-doctor-primary" size={14} /> : <FaChevronDown className="text-doctor-primary" size={14} />}
          </button>

          {/* Subcategories Collapsible Content */}
          <div className={`overflow-hidden transition-all duration-300 ease-in-out ${openDepartmentIndex === deptIndex ? 'max-h-[500px] opacity-100' : 'max-h-0 opacity-0'}`}>
            {department.subcategories.length === 0 ? (
              <p className="px-4 py-4 text-sm text-gray-500 font-body">لم يتم إضافة أقسام فرعية بعد</p>
            ) : (
              <ul className="px-2 pb-2 space-y-2 py-4">
                {department.subcategories.map((subcategory, subIndex) => (
                  <li key={subcategory.id} className="bg-white rounded-md shadow-sm">
                    {/* Subcategory Toggle */}
                    <button
                      onClick={() => toggleSubcategory(subIndex)}
                      className="w-full flex justify-between items-center py-2 px-3 text-gray-700 hover:bg-gray-50 focus:bg-gray-50 transition-colors duration-200 focus:outline-none focus:ring-1 focus:ring-blue-200"
                    >
                      <span className="text-xs font-medium font-body">{subcategory.title}</span>
                      {openSubcategoryIndex === subIndex ? <FaChevronUp className="text-doctor-primary" size={12} /> : <FaChevronDown className="text-doctor-primary" size={12} />}
                    </button>

                    {/* Posts Collapsible Content */}
                    <div className={`overflow-hidden transition-all duration-300 ease-in-out ${openSubcategoryIndex === subIndex ? 'max-h-48 opacity-100' : 'max-h-0 opacity-0'}`}>
                      {subcategory.posts.length === 0 ? (
                        <p className="px-4 py-4 text-xs text-gray-500 font-body">لم يتم إضافة منشورات بعد</p>
                      ) : (
                        <ul className="px-4 pb-2 space-y-1 py-4">
                          {subcategory.posts.map((post) => (
                            <li key={post.id}>
                              <Link
                                href={post.link}
                                className="flex items-center gap-1 text-xs text-gray-600 font-body py-1 px-2 rounded-md hover:bg-blue-50 hover:text-doctor-primary transition-all duration-200"
                              >
                                <MdKeyboardArrowLeft size={15} />
                                {post.title}
                              </Link>
                            </li>
                          ))}
                        </ul>
                      )}
                    </div>
                  </li>
                ))}
              </ul>
            )}
          </div>
        </li>
      ))}
    </ul>
  );
};

export default DepartmentsAsideClient;
