import { Workplace } from '@/app/(private)/dashboard/workplaces/workplace.schema';
import { FaMapMarkerAlt, FaPhoneAlt, FaEnvelope, FaMoneyBillWave, FaLink } from 'react-icons/fa';

async function fetchWorkplaceItems(): Promise<Workplace[]> {
  const response = await fetch(`${process.env.NEXT_PUBLIC_SERVER_URL}/workplace/list`, {
    next: { revalidate: 30 }, // Revalidate every 30 seconds
  });
  if (!response.ok) {
    throw new Error('Failed to fetch workplace items');
  }
  return response.json();
}

const ContactUsSection = async () => {
  try {
    const workplaceItems = await fetchWorkplaceItems();

    return (
      <section className="py-12 bg-white border border-neutral-200 rounded-lg">
        <div className="mx-auto w-full max-w-7xl px-4 sm:px-6 lg:px-8">
          <h2 className="text-3xl md:text-4xl font-bold font-heading text-doctor-primary mb-6">تواصل معنا</h2>
          <p className="mb-8 text-base text-gray-500 font-body max-w-2xl md:mb-12">تواصلوا معنا للحصول على أفضل الخدمات الطبية في أفضل المستشفيات والمراكز الطبية.</p>
          <div className="grid gap-6 sm:grid-cols-2 lg:grid-cols-3">
            {workplaceItems.length > 0 ? (
              workplaceItems.map((info) => (
                <div key={info.id} className="bg-white rounded-lg p-6 flex flex-col gap-4 border border-gray-200 hover:shadow-md transition-all duration-300">
                  <h3 className="text-lg font-semibold text-gray-800 font-heading">{info.name}</h3>
                  <div className="flex items-start gap-3">
                    <FaMapMarkerAlt className="text-cyan-500 mt-1" size={20} />
                    <p className="text-sm text-gray-600 font-body flex-1">{info.address}</p>
                  </div>
                  <div className="flex items-center gap-3">
                    <FaPhoneAlt className="text-cyan-500" size={20} />
                    <p className="text-sm text-gray-600 font-body">{info.phone_1}</p>
                  </div>
                  {info.phone_2 && (
                    <div className="flex items-center gap-3">
                      <FaPhoneAlt className="text-cyan-500" size={20} />
                      <p className="text-sm text-gray-600 font-body">{info.phone_2}</p>
                    </div>
                  )}
                  <div className="flex items-center gap-3">
                    <FaEnvelope className="text-cyan-500" size={20} />
                    <a href={`mailto:${info.email}`} target="_blank" rel="noopener noreferrer" className="text-sm text-cyan-500 hover:underline font-body">
                      {info.email}
                    </a>
                  </div>
                  {info.url && (
                    <div className="flex items-center gap-3">
                      <FaLink className="text-cyan-500" size={20} />
                      <a href={info.url} target="_blank" rel="noopener noreferrer" className="text-sm text-cyan-500 hover:underline font-body">
                        {info.url}
                      </a>
                    </div>
                  )}
                  {info.examine_price && (
                    <div className="flex items-center gap-3">
                      <FaMoneyBillWave className="text-cyan-500" size={20} />
                      <p className="text-sm text-gray-600 font-body">سعر الكشف: {info.examine_price}</p>
                    </div>
                  )}
                </div>
              ))
            ) : (
              <p className="text-center text-gray-600 col-span-full">لا توجد أماكن عمل متاحة حاليًا</p>
            )}
          </div>
        </div>
      </section>
    );
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
  } catch (error) {
    return (
      <section className="py-12 bg-white border border-neutral-200 rounded-lg">
        <div className="mx-auto w-full max-w-7xl px-4 sm:px-6 lg:px-8">
          <h2 className="text-3xl md:text-4xl font-bold font-heading text-doctor-primary mb-6">تواصل معنا</h2>
          <p className="text-center text-red-500">حدث خطأ أثناء جلب البيانات. حاول مرة أخرى لاحقًا.</p>
        </div>
      </section>
    );
  }
};

export default ContactUsSection;
