'use client';

import { zodResolver } from '@hookform/resolvers/zod';
import { useForm } from 'react-hook-form';
import { useState } from 'react';
import { z } from 'zod';

// Define the schema using Zod
const contactFormSchema = z.object({
  full_name: z.string().min(1, 'الاسم مطلوب').max(100, 'الاسم طويل جدًا'),
  mail: z.string().email('البريد الإلكتروني غير صالح').min(1, 'البريد الإلكتروني مطلوب'),
  phone_number: z
    .string()
    .min(10, 'رقم الهاتف يجب أن يكون 10 أرقام على الأقل')
    .max(15, 'رقم الهاتف طويل جدًا')
    .regex(/^[0-9]+$/, 'رقم الهاتف يجب أن يحتوي على أرقام فقط'),
  message: z.string().min(1, 'الرسالة مطلوبة').max(500, 'الرسالة طويلة جدًا'),
});

// Infer the TypeScript type from the schema
type ContactFormValues = z.infer<typeof contactFormSchema>;

const ContactForm = () => {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [successMessage, setSuccessMessage] = useState<string | null>(null);
  const [errorMessage, setErrorMessage] = useState<string | null>(null);

  // Initialize React Hook Form with Zod resolver
  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
  } = useForm<ContactFormValues>({
    resolver: zodResolver(contactFormSchema),
    defaultValues: {
      full_name: '',
      mail: '',
      phone_number: '',
      message: '',
    },
  });

  // Handle form submission
  const onSubmit = async (data: ContactFormValues) => {
    setIsSubmitting(true);
    setSuccessMessage(null);
    setErrorMessage(null);

    try {
      const response = await fetch(`${process.env.NEXT_PUBLIC_SERVER_URL}/contactUs/create`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(data),
      });

      if (response.status === 201) {
        setSuccessMessage('تم إرسال رسالتك بنجاح! سنتواصل معك قريبًا.');
        reset(); // Clear the form on success
      } else {
        const errorData = await response.json();
        setErrorMessage(errorData.message || 'حدث خطأ أثناء إرسال الرسالة. حاول مرة أخرى.');
      }
      // eslint-disable-next-line @typescript-eslint/no-unused-vars
    } catch (error) {
      setErrorMessage('فشل الاتصال بالخادم. تأكد من اتصالك بالإنترنت وحاول مجددًا.');
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <div className="p-6 rounded-lg border border-neutral-200 bg-white">
      <h3 className="text-2xl font-bold text-doctor-dark mb-6 font-heading">اترك رسالتك</h3>

      {/* Success Message */}
      {successMessage && (
        <div className="mb-4 p-3 bg-green-100 text-green-800 rounded-md flex items-center gap-2">
          <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
          </svg>
          {successMessage}
        </div>
      )}

      {/* Error Message */}
      {errorMessage && (
        <div className="mb-4 p-3 bg-red-100 text-red-800 rounded-md flex items-center gap-2">
          <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M6 18L18 6M6 6l12 12" />
          </svg>
          {errorMessage}
        </div>
      )}

      <form onSubmit={handleSubmit(onSubmit)} className="space-y-4" dir="rtl">
        {/* Full Name */}
        <div>
          <label htmlFor="full_name" className="block text-sm font-medium text-gray-700 mb-1">
            الاسم
          </label>
          <input
            type="text"
            id="full_name"
            {...register('full_name')}
            className={`w-full px-3 py-2 border rounded-md focus:outline-none focus:ring-2 focus:ring-doctor-primary ${errors.full_name ? 'border-red-500' : 'border-gray-300'}`}
            placeholder="أدخل اسمك"
            disabled={isSubmitting}
          />
          {errors.full_name && <p className="text-red-500 text-xs mt-1">{errors.full_name.message}</p>}
        </div>

        {/* Email */}
        <div>
          <label htmlFor="mail" className="block text-sm font-medium text-gray-700 mb-1">
            البريد الإلكتروني
          </label>
          <input
            type="email"
            id="mail"
            {...register('mail')}
            className={`w-full px-3 py-2 border rounded-md focus:outline-none focus:ring-2 focus:ring-doctor-primary ${errors.mail ? 'border-red-500' : 'border-gray-300'}`}
            placeholder="أدخل بريدك الإلكتروني"
            disabled={isSubmitting}
          />
          {errors.mail && <p className="text-red-500 text-xs mt-1">{errors.mail.message}</p>}
        </div>

        {/* Phone Number */}
        <div>
          <label htmlFor="phone_number" className="block text-sm font-medium text-gray-700 mb-1">
            رقم الهاتف
          </label>
          <input
            type="text" // Using text instead of tel as requested
            id="phone_number"
            {...register('phone_number')}
            className={`w-full px-3 py-2 border rounded-md focus:outline-none focus:ring-2 focus:ring-doctor-primary ${errors.phone_number ? 'border-red-500' : 'border-gray-300'}`}
            placeholder="أدخل رقم هاتفك"
            disabled={isSubmitting}
          />
          {errors.phone_number && <p className="text-red-500 text-xs mt-1">{errors.phone_number.message}</p>}
        </div>

        {/* Message */}
        <div>
          <label htmlFor="message" className="block text-sm font-medium text-gray-700 mb-1">
            الرسالة
          </label>
          <textarea
            id="message"
            {...register('message')}
            rows={4}
            className={`w-full px-3 py-2 border rounded-md focus:outline-none focus:ring-2 focus:ring-doctor-primary resize-none ${
              errors.message ? 'border-red-500' : 'border-gray-300'
            }`}
            placeholder="اكتب رسالتك هنا..."
            disabled={isSubmitting}
          />
          {errors.message && <p className="text-red-500 text-xs mt-1">{errors.message.message}</p>}
        </div>

        {/* Submit Button */}
        <button
          type="submit"
          disabled={isSubmitting}
          className="w-full bg-doctor-primary text-white px-6 py-3 rounded-lg hover:bg-doctor-dark transition-all duration-300 font-body text-base font-medium flex items-center justify-center gap-2 disabled:opacity-50"
        >
          {isSubmitting ? (
            <>
              <svg className="animate-spin w-5 h-5" fill="none" viewBox="0 0 24 24">
                <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" />
                <path
                  className="opacity-75"
                  fill="currentColor"
                  d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                />
              </svg>
              جارٍ الإرسال...
            </>
          ) : (
            'إرسال الرسالة'
          )}
        </button>
      </form>
    </div>
  );
};

export default ContactForm;
