import Link from 'next/link';
import Image from 'next/image';

// Define the BlogPost interface based on the backend response
interface BlogPost {
  id: number;
  title: string;
  description: string | null;
  thumbnail: { url: string }[];
  category_name: string;
  link: string;
}

// Fetch latest posts with ISR
async function fetchLatestPosts(): Promise<BlogPost[]> {
  const response = await fetch(`${process.env.NEXT_PUBLIC_SERVER_URL}/post/latest-posts?limit=6`, {
    next: { revalidate: 60 }, // Revalidate every 60 seconds
  });

  if (!response.ok) {
    throw new Error('Failed to fetch latest posts');
  }

  const posts = await response.json();

  // Map the backend response to include the link field
  return posts.map((post: BlogPost) => ({
    id: post.id,
    title: post.title,
    description: post.description,
    thumbnail: post.thumbnail, // Assuming thumbnail is an array of objects with url
    category_name: post.category_name,
    link: `/posts/${post.id}`, // Construct link based on post ID
  }));
}

export default async function ArticlesSection() {
  try {
    const blogPosts = await fetchLatestPosts();

    return (
      <section className="py-12 bg-white border border-neutral-200 rounded-lg">
        {/* Container */}
        <div className="mx-auto w-full max-w-7xl px-4 sm:px-6 lg:px-8">
          {/* Title */}
          <h2 className="text-3xl md:text-4xl font-bold font-heading text-doctor-primary mb-6">آخر الأخبار والمقالات</h2>
          <p className="mb-8 text-base text-gray-500 font-body max-w-2xl md:mb-12">اطلع على أحدث النصائح والمقالات حول الأشعة التداخلية والرعاية الصحية المتقدمة.</p>

          {/* Grid Content */}
          {blogPosts.length === 0 ? (
            <p className="text-gray-500 text-sm">لم يتم إضافة مقالات بعد</p>
          ) : (
            <div className="grid gap-6 sm:grid-cols-2 lg:grid-cols-2">
              {blogPosts.map((post) => (
                <Link key={post.id} href={post.link} className="group flex bg-white rounded-xl hover:shadow-sm transition-all duration-300 border border-gray-200 p-4 items-center">
                  {/* Image Section */}
                  <div className="relative w-28 h-28">
                    <Image src={post.thumbnail?.[0]?.url} alt={post.title} fill className="object-cover rounded-lg transition-transform duration-300 group-hover:scale-102" />
                  </div>
                  {/* Text Section */}
                  <div className="flex-1 flex flex-col gap-2 pr-4">
                    <div className="inline-block w-fit rounded-full bg-cyan-100 px-3 py-1 text-xs font-semibold text-cyan-500">{post.category_name}</div>
                    <h3 className="text-lg font-bold text-doctor-dark font-heading line-clamp-2">{post.title}</h3>
                    <p className="text-sm text-gray-600 font-body line-clamp-2">{post.description || 'لا يوجد وصف متاح'}</p>
                  </div>
                </Link>
              ))}
            </div>
          )}
        </div>
      </section>
    );
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
  } catch (error) {
    return (
      <section className="py-12 bg-white border border-neutral-200 rounded-lg">
        <div className="mx-auto w-full max-w-7xl px-4 sm:px-6 lg:px-8">
          <h2 className="text-3xl md:text-4xl font-bold font-heading text-doctor-primary mb-6">آخر الأخبار والمقالات</h2>
          <p className="mb-8 text-base text-gray-500 font-body max-w-2xl md:mb-12">اطلع على أحدث النصائح والمقالات حول الأشعة التداخلية والرعاية الصحية المتقدمة.</p>
          <p className="text-red-500 text-sm">حدث خطأ أثناء جلب البيانات. حاول مرة أخرى لاحقًا.</p>
        </div>
      </section>
    );
  }
}
