import React from 'react';
import Link from 'next/link';
import Image from 'next/image';
import { ApiResponse } from '@/app/(private)/dashboard/user/user.schema';
import { FaImage } from 'react-icons/fa';

// Fetch public user details with ISR
async function fetchUserPublicDetails(): Promise<ApiResponse> {
  const response = await fetch(`${process.env.NEXT_PUBLIC_SERVER_URL}/user/me/public`, {
    next: { revalidate: 30 }, // Revalidate every 30 seconds
  });
  if (!response.ok) {
    throw new Error('Failed to fetch user public details');
  }
  return response.json();
}

export default async function AboutSection() {
  try {
    const userData = await fetchUserPublicDetails();
    const user = userData.data;

    return (
      <section className="py-12 bg-white border border-neutral-200 rounded-lg">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          {/* Content and Image Grid */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
            {/* Text Content */}
            <div className="space-y-6">
              <h2 className="text-xl md:text-2xl font-semibold text-doctor-secondary font-heading">نبذة عنــــــــــــا</h2>
              <p className="text-gray-600 text-sm md:text-base font-body leading-relaxed">
                {user.about_dr || 'لا توجد معلومات متاحة عن الطبيب في الوقت الحالي. يرجى التحقق لاحقًا للحصول على تفاصيل دقيقة.'}
              </p>

              <Link
                href="/about"
                className="inline-block bg-doctor-primary text-white px-6 py-3 rounded-lg hover:bg-doctor-dark transition-colors duration-300 font-body text-base md:text-lg font-medium"
              >
                تعرف على المزيد
              </Link>
            </div>

            {/* Image */}
            <div className="relative w-full h-64 md:h-80 lg:h-96">
              {user.doctor_pic?.[0]?.url ? (
                <Image src={user.doctor_pic?.[0]?.url} alt={user.user_name} fill className="object-contain" priority={true} />
              ) : (
                <div className="w-full h-full bg-gray-100 rounded-lg flex items-center justify-center text-gray-500 my-auto">
                  <FaImage className="text-9xl" />
                </div>
              )}
            </div>
          </div>
        </div>
      </section>
    );
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
  } catch (error) {
    return (
      <section className="py-12 bg-white border border-neutral-200 rounded-lg">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 items-center">
            <div className="space-y-6">
              <h2 className="text-xl md:text-2xl font-semibold text-doctor-secondary font-heading">نبذة عنــــــــــــا</h2>
              <p className="text-red-500">حدث خطأ أثناء جلب البيانات. حاول مرة أخرى لاحقًا.</p>
            </div>
            <div className="relative w-full h-64 md:h-80 lg:h-96">
              <Image src="/doctor.webp" alt="Doctor in clinic" fill className="object-contain" priority={true} />
            </div>
          </div>
        </div>
      </section>
    );
  }
}
