import Cookies from 'js-cookie';
import { CreateWorkplaceFormValues, UpdateWorkplaceFormValues, Workplace } from './workplace.schema';

const API_URL = process.env.NEXT_PUBLIC_SERVER_URL || 'http://localhost:3000';

// Helper function to get the auth token from cookies
const getAuthToken = () => {
  if (typeof window !== 'undefined') {
    return Cookies.get('dr_auth_token');
  }
  return null;
};

// List all workplace items (no pagination or filters)
export const listWorkplace = async (): Promise<Workplace[]> => {
  const response = await fetch(`${API_URL}/workplace/list`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Workplace[];
};

// Create a new workplace item
export const createWorkplace = async (data: CreateWorkplaceFormValues): Promise<Workplace> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/workplace/create`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Workplace;
};

// Get workplace item details by ID
export const getWorkplaceDetails = async (id: number): Promise<Workplace> => {
  const response = await fetch(`${API_URL}/workplace/${id}`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Workplace;
};

// Update workplace item
export const updateWorkplace = async (id: number, data: UpdateWorkplaceFormValues): Promise<Workplace> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/workplace/${id}`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Workplace;
};

// Delete multiple workplace items
export const deleteWorkplace = async (ids: number[]): Promise<void> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/workplace/delete-many`, {
    method: 'DELETE',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify({ ids }),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result;
};
