/* eslint-disable @typescript-eslint/no-explicit-any */

'use client';

import { useState } from 'react';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { FaEdit, FaTrash, FaPlus, FaMapMarkerAlt, FaPhoneAlt, FaEnvelope, FaMoneyBillWave, FaLink } from 'react-icons/fa';
import { listWorkplace, createWorkplace, updateWorkplace, deleteWorkplace, getWorkplaceDetails } from './workplace.service';
import { CreateWorkplaceFormValues, UpdateWorkplaceFormValues } from './workplace.schema';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { createWorkplaceFormSchema, updateWorkplaceFormSchema } from './workplace.schema';

const Workplace = () => {
  const queryClient = useQueryClient();
  const [selectedIds, setSelectedIds] = useState<number[]>([]);
  const [isCreateModalOpen, setIsCreateModalOpen] = useState(false);
  const [isEditModalOpen, setIsEditModalOpen] = useState(false);
  const [editingWorkplaceId, setEditingWorkplaceId] = useState<number | null>(null);
  const [formError, setFormError] = useState<string | null>(null);

  // Fetch all workplace items
  const {
    data: workplaceItems,
    isLoading,
    error,
  } = useQuery({
    queryKey: ['workplace'],
    queryFn: listWorkplace,
  });

  // Form setup for create
  const createForm = useForm<CreateWorkplaceFormValues>({
    resolver: zodResolver(createWorkplaceFormSchema),
    defaultValues: {
      name: '',
      url: '',
      address: '',
      phone_1: '',
      phone_2: '',
      email: '',
      examine_price: '',
    },
  });

  // Form setup for update
  const updateForm = useForm<UpdateWorkplaceFormValues>({
    resolver: zodResolver(updateWorkplaceFormSchema),
    defaultValues: {
      name: '',
      url: '',
      address: '',
      phone_1: '',
      phone_2: '',
      email: '',
      examine_price: '',
    },
  });

  // Create mutation
  const createMutation = useMutation({
    mutationFn: createWorkplace,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['workplace'] });
      setIsCreateModalOpen(false);
      setFormError(null);
      createForm.reset();
    },
    onError: (err: any) => {
      console.error('Create workplace error:', err);
      setFormError('فشل في إنشاء مكان العمل: ' + (err.message || 'خطأ غير معروف'));
    },
  });

  // Update mutation
  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: UpdateWorkplaceFormValues }) => updateWorkplace(id, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['workplace'] });
      setIsEditModalOpen(false);
      setEditingWorkplaceId(null);
      setFormError(null);
      updateForm.reset();
    },
    onError: (err: any) => {
      console.error('Update workplace error:', err);
      setFormError('فشل في تحديث مكان العمل: ' + (err.message || 'خطأ غير معروف'));
    },
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: deleteWorkplace,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['workplace'] });
      setSelectedIds([]);
    },
    onError: (err: any) => {
      console.error('Delete workplace error:', err);
      setFormError('فشل في حذف أماكن العمل: ' + (err.message || 'خطأ غير معروف'));
    },
  });

  const handleSelectWorkplace = (id: number) => {
    setSelectedIds((prev) => (prev.includes(id) ? prev.filter((item) => item !== id) : [...prev, id]));
  };

  const handleCreateSubmit = (data: CreateWorkplaceFormValues) => {
    createMutation.mutate(data);
  };

  const handleEditClick = async (id: number) => {
    try {
      const workplace = await getWorkplaceDetails(id);
      setEditingWorkplaceId(id);
      updateForm.reset({
        name: workplace.name,
        url: workplace.url || '',
        address: workplace.address,
        phone_1: workplace.phone_1,
        phone_2: workplace.phone_2 || '',
        email: workplace.email,
        examine_price: workplace.examine_price || '',
      });
      setIsEditModalOpen(true);
      setFormError(null);
    } catch (err) {
      console.error('Error fetching workplace details:', err);
      setFormError('فشل في جلب تفاصيل مكان العمل');
    }
  };

  const handleUpdateSubmit = (data: UpdateWorkplaceFormValues) => {
    if (editingWorkplaceId) {
      updateMutation.mutate({ id: editingWorkplaceId, data });
    }
  };

  const handleDeleteSelected = () => {
    if (selectedIds.length > 0) {
      deleteMutation.mutate(selectedIds);
    }
  };

  const openCreateModal = () => {
    createForm.reset();
    setFormError(null);
    setIsCreateModalOpen(true);
  };

  return (
    <div className="px-2 py-4 min-h-screen" dir="rtl">
      <h2 className="text-xl sm:text-2xl font-bold mb-6 text-doctor-dark font-heading text-start">إدارة أماكن العمل</h2>

      {/* Actions */}
      <div className="flex justify-between mb-6">
        <button onClick={openCreateModal} className="px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary flex items-center gap-2">
          <FaPlus /> إضافة مكان عمل
        </button>
        {selectedIds.length > 0 && (
          <button
            onClick={handleDeleteSelected}
            className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 flex items-center gap-2"
            disabled={deleteMutation.isPending}
          >
            <FaTrash /> حذف ({selectedIds.length})
          </button>
        )}
      </div>

      {/* Workplace List */}
      {isLoading ? (
        <div className="text-center py-8 text-gray-600 text-lg">جارٍ التحميل...</div>
      ) : error ? (
        <div className="text-center py-8 text-red-500 text-lg">حدث خطأ أثناء جلب البيانات: {JSON.stringify(error)}</div>
      ) : workplaceItems && workplaceItems.length > 0 ? (
        <div className="grid gap-6 sm:grid-cols-2 lg:grid-cols-3">
          {workplaceItems.map((workplace) => (
            <div
              key={workplace.id}
              className={`bg-white rounded-lg p-6 flex flex-col gap-4 border border-gray-200 hover:shadow-md transition-all duration-300 ${
                selectedIds.includes(workplace.id) ? 'border-doctor-dark border-2' : ''
              }`}
            >
              {/* Name */}
              <div className="flex items-center justify-between">
                <h3 className="text-lg font-semibold text-gray-800 font-heading">{workplace.name}</h3>
                <div className="flex gap-3">
                  <button onClick={() => handleEditClick(workplace.id)} className="text-doctor-dark hover:text-doctor-primary">
                    <FaEdit size={20} />
                  </button>
                  <input
                    type="checkbox"
                    className="h-5 w-5 text-blue-600 rounded focus:ring-blue-500"
                    checked={selectedIds.includes(workplace.id)}
                    onChange={() => handleSelectWorkplace(workplace.id)}
                  />
                </div>
              </div>

              {/* Address */}
              <div className="flex items-start gap-3">
                <FaMapMarkerAlt className="text-cyan-500 mt-1" size={20} />
                <p className="text-sm text-gray-600 font-body flex-1">{workplace.address}</p>
              </div>

              {/* Phone 1 */}
              <div className="flex items-center gap-3">
                <FaPhoneAlt className="text-cyan-500" size={20} />
                <p className="text-sm text-gray-600 font-body">{workplace.phone_1}</p>
              </div>

              {/* Phone 2 (Optional) */}
              {workplace.phone_2 && (
                <div className="flex items-center gap-3">
                  <FaPhoneAlt className="text-cyan-500" size={20} />
                  <p className="text-sm text-gray-600 font-body">{workplace.phone_2}</p>
                </div>
              )}

              {/* Email */}
              <div className="flex items-center gap-3">
                <FaEnvelope className="text-cyan-500" size={20} />
                <a href={`mailto:${workplace.email}`} target="_blank" rel="noopener noreferrer" className="text-sm text-cyan-500 hover:underline font-body">
                  {workplace.email}
                </a>
              </div>

              {/* URL (Optional) */}
              {workplace.url && (
                <div className="flex items-center gap-3">
                  <FaLink className="text-cyan-500" size={20} />
                  <a href={workplace.url} target="_blank" rel="noopener noreferrer" className="text-sm text-cyan-500 hover:underline font-body">
                    {workplace.url}
                  </a>
                </div>
              )}

              {/* Examine Price (Optional) */}
              {workplace.examine_price && (
                <div className="flex items-center gap-3">
                  <FaMoneyBillWave className="text-cyan-500" size={20} />
                  <p className="text-sm text-gray-600 font-body">سعر الكشف: {workplace.examine_price}</p>
                </div>
              )}
            </div>
          ))}
        </div>
      ) : (
        <div className="text-center py-8 text-gray-600 text-lg">لا توجد أماكن عمل</div>
      )}

      {/* Create Modal */}
      {isCreateModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsCreateModalOpen(false)} />
          <div className="relative bg-white p-6 sm:p-8 rounded-xl w-full max-w-md shadow-2xl">
            <h3 className="text-xl font-bold mb-4 text-doctor-dark">إضافة مكان عمل جديد</h3>
            {formError && <p className="text-red-500 text-sm mb-4">{formError}</p>}
            <form onSubmit={createForm.handleSubmit(handleCreateSubmit)} className="space-y-4" dir="rtl">
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الاسم</label>
                <input {...createForm.register('name')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {createForm.formState.errors.name && <p className="text-red-500 text-sm mt-1">{createForm.formState.errors.name.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">رابط الموقع</label>
                <input {...createForm.register('url')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {createForm.formState.errors.url && <p className="text-red-500 text-sm mt-1">{createForm.formState.errors.url.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">العنوان</label>
                <textarea {...createForm.register('address')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {createForm.formState.errors.address && <p className="text-red-500 text-sm mt-1">{createForm.formState.errors.address.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">رقم الهاتف الأول</label>
                <input {...createForm.register('phone_1')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {createForm.formState.errors.phone_1 && <p className="text-red-500 text-sm mt-1">{createForm.formState.errors.phone_1.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">رقم الهاتف الثاني</label>
                <input {...createForm.register('phone_2')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {createForm.formState.errors.phone_2 && <p className="text-red-500 text-sm mt-1">{createForm.formState.errors.phone_2.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">البريد الإلكتروني</label>
                <input {...createForm.register('email')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {createForm.formState.errors.email && <p className="text-red-500 text-sm mt-1">{createForm.formState.errors.email.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">سعر الكشف</label>
                <input {...createForm.register('examine_price')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {createForm.formState.errors.examine_price && <p className="text-red-500 text-sm mt-1">{createForm.formState.errors.examine_price.message}</p>}
              </div>
              <div className="flex gap-3">
                <button
                  type="submit"
                  disabled={createMutation.isPending || !createForm.formState.isValid}
                  className="flex-1 px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary disabled:opacity-50"
                >
                  {createMutation.isPending ? 'جارٍ...' : 'إضافة'}
                </button>
                <button type="button" onClick={() => setIsCreateModalOpen(false)} className="flex-1 px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400">
                  إلغاء
                </button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* Edit Modal */}
      {isEditModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsEditModalOpen(false)} />
          <div className="relative bg-white p-6 sm:p-8 rounded-xl w-full max-w-md shadow-2xl">
            <h3 className="text-xl font-bold mb-4 text-doctor-dark">تعديل مكان العمل</h3>
            {formError && <p className="text-red-500 text-sm mb-4">{formError}</p>}
            <form onSubmit={updateForm.handleSubmit(handleUpdateSubmit)} className="space-y-4" dir="rtl">
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الاسم</label>
                <input {...updateForm.register('name')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {updateForm.formState.errors.name && <p className="text-red-500 text-sm mt-1">{updateForm.formState.errors.name.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">رابط الموقع</label>
                <input {...updateForm.register('url')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {updateForm.formState.errors.url && <p className="text-red-500 text-sm mt-1">{updateForm.formState.errors.url.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">العنوان</label>
                <textarea {...updateForm.register('address')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {updateForm.formState.errors.address && <p className="text-red-500 text-sm mt-1">{updateForm.formState.errors.address.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">رقم الهاتف الأول</label>
                <input {...updateForm.register('phone_1')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {updateForm.formState.errors.phone_1 && <p className="text-red-500 text-sm mt-1">{updateForm.formState.errors.phone_1.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">رقم الهاتف الثاني</label>
                <input {...updateForm.register('phone_2')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {updateForm.formState.errors.phone_2 && <p className="text-red-500 text-sm mt-1">{updateForm.formState.errors.phone_2.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">البريد الإلكتروني</label>
                <input {...updateForm.register('email')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {updateForm.formState.errors.email && <p className="text-red-500 text-sm mt-1">{updateForm.formState.errors.email.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">سعر الكشف</label>
                <input {...updateForm.register('examine_price')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {updateForm.formState.errors.examine_price && <p className="text-red-500 text-sm mt-1">{updateForm.formState.errors.examine_price.message}</p>}
              </div>
              <div className="flex gap-3">
                <button
                  type="submit"
                  disabled={updateMutation.isPending || !updateForm.formState.isValid}
                  className="flex-1 px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary disabled:opacity-50"
                >
                  {updateMutation.isPending ? 'جارٍ...' : 'حفظ'}
                </button>
                <button type="button" onClick={() => setIsEditModalOpen(false)} className="flex-1 px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400">
                  إلغاء
                </button>
              </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
};

export default Workplace;
