import Cookies from 'js-cookie';
import { UpdateUserFormValues, UpdatePasswordFormValues, ApiResponse, UploadFileResponse, FileMetadata } from './user.schema';

const API_URL = process.env.NEXT_PUBLIC_SERVER_URL || 'http://localhost:7000';

// Helper function to get the auth token from cookies
const getAuthToken = () => {
  if (typeof window !== 'undefined') {
    return Cookies.get('dr_auth_token');
  }
  return null;
};

// Fetch logged-in user details (private endpoint)
export const getUserDetails = async (): Promise<ApiResponse> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/user/me`, {
    method: 'GET',
    headers: {
      Authorization: `Bearer ${token}`,
    },
  });

  const result = await response.json();

  if (!response.ok) {
    throw result;
  }

  return result as ApiResponse;
};

// Fetch public user details (public endpoint)
export const getUserPublicDetails = async (): Promise<ApiResponse> => {
  const response = await fetch(`${API_URL}/user/me/public`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as ApiResponse;
};

// Update user details (user_name, about_dr, logo, doctor_pic, login_key)
export const updateUser = async (data: UpdateUserFormValues & { logo?: FileMetadata[]; doctor_pic?: FileMetadata[] }): Promise<ApiResponse> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/user/me`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as ApiResponse;
};

// Update user password
export const updateUserPassword = async (data: UpdatePasswordFormValues): Promise<ApiResponse> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/user/me/password`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as ApiResponse;
};

// Upload file (logo or doctor_pic)
export const uploadFile = async (file: File): Promise<UploadFileResponse> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const formData = new FormData();
  formData.append('files', file);

  const response = await fetch(`${API_URL}/multerUploader/upload`, {
    method: 'POST',
    headers: {
      Authorization: `Bearer ${token}`,
    },
    body: formData,
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as UploadFileResponse;
};

// Remove file (delete old image from server)
export const removeFile = async (fileUrl: string): Promise<void> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  await fetch(`${API_URL}/multerUploader/remove`, {
    method: 'DELETE',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify({
      files: [{ url: fileUrl }],
    }),
  });
};
