import { z } from 'zod';

// Define the FileMetadata type (for logo and doctor_pic)
export interface FileMetadata {
  url: string;
  filename: string;
}

// Define the User type based on the backend schema
export interface User {
  id: number;
  user_name: string;
  email: string;
  about_dr: string | null;
  logo: FileMetadata[] | null;
  doctor_pic: FileMetadata[] | null;
  login_key: string | null; // Added login_key
  createdAt: string;
  updatedAt: string;
}

// Base schema for shared fields in user forms
const baseUserSchema = z.object({
  user_name: z.string().min(1, 'اسم المستخدم مطلوب').max(255, 'اسم المستخدم يجب ألا يتجاوز 255 حرفًا').optional(),
  email: z.string().email('البريد الإلكتروني غير صالح').optional(),
  about_dr: z.string().max(1000, 'الوصف يجب ألا يتجاوز 1000 حرف').optional().nullable(),
  login_key: z.string().max(255, 'مفتاح الدخول يجب ألا يتجاوز 255 حرفًا').optional().nullable(), // Added login_key
});

// Schema for updating user details (user_name, about_dr, login_key)
export const updateUserFormSchema = baseUserSchema;

// Schema for updating user password
export const updatePasswordFormSchema = z.object({
  oldPassword: z.string().min(1, 'كلمة المرور القديمة مطلوبة'),
  newPassword: z.string().min(5, 'كلمة المرور الجديدة يجب أن تكون 5 أحرف على الأقل'),
});

export type UpdateUserFormValues = z.infer<typeof updateUserFormSchema>;
export type UpdatePasswordFormValues = z.infer<typeof updatePasswordFormSchema>;

// Schema for file upload response
export const uploadFileSchema = z.object({
  message: z.string(),
  data: z.object({
    files: z.array(
      z.object({
        url: z.string(),
        filename: z.string(),
      }),
    ),
  }),
});

export type UploadFileResponse = z.infer<typeof uploadFileSchema>;

// Schema for API response (for user endpoints)
export const apiResponseSchema = z.object({
  statusCode: z.number(),
  data: z.object({
    id: z.number(),
    user_name: z.string(),
    email: z.string(),
    about_dr: z.string().nullable(),
    logo: z.array(z.object({ url: z.string(), filename: z.string() })).nullable(),
    doctor_pic: z.array(z.object({ url: z.string(), filename: z.string() })).nullable(),
    login_key: z.string().nullable(), // Added login_key
    createdAt: z.string(),
    updatedAt: z.string(),
  }),
});

export type ApiResponse = z.infer<typeof apiResponseSchema>;
