/* eslint-disable @typescript-eslint/no-explicit-any */
/* eslint-disable @typescript-eslint/no-unused-vars */
'use client';

import { handleBackendError } from '@/shared/api/errorHandler';
import { zodResolver } from '@hookform/resolvers/zod';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import Image from 'next/image';
import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { FaImage, FaLock, FaStar, FaTrash, FaUser } from 'react-icons/fa';
import { FileMetadata, updatePasswordFormSchema, UpdatePasswordFormValues, updateUserFormSchema, UpdateUserFormValues } from './user.schema';
import { getUserDetails, updateUser, updateUserPassword, uploadFile, removeFile } from './user.service';
import { createReview, deleteManyReviews, listGlobalReviews, updateReview } from '../reviews/reviews.service';
import { createReviewFormSchema, CreateReviewFormValues, Review, updateReviewFormSchema, UpdateReviewFormValues } from '../reviews/reviews.schema';

const UserProfile = () => {
  const queryClient = useQueryClient();
  const [isDetailsModalOpen, setIsDetailsModalOpen] = useState(false);
  const [isPasswordModalOpen, setIsPasswordModalOpen] = useState(false);
  const [isReviewModalOpen, setIsReviewModalOpen] = useState(false);
  const [isEditReviewModalOpen, setIsEditReviewModalOpen] = useState(false);
  const [selectedReview, setSelectedReview] = useState<number | null>(null);
  const [selectedReviews, setSelectedReviews] = useState<number[]>([]);
  const [isImageLoading, setIsImageLoading] = useState<{ logo: boolean; doctor_pic: boolean }>({ logo: false, doctor_pic: false });
  const [reviewFilter, setReviewFilter] = useState<'all' | 'active' | 'in_active'>('all'); // New filter state

  // Fetch user details
  const {
    data: userResponse,
    isLoading,
    error,
  } = useQuery({
    queryKey: ['user'],
    queryFn: getUserDetails,
  });
  const user = userResponse?.data;

  // Fetch global reviews with filter
  const {
    data: reviews,
    isLoading: reviewsLoading,
    error: reviewsError,
  } = useQuery({
    queryKey: ['globalReviews', reviewFilter],
    queryFn: () => {
      if (reviewFilter === 'all') {
        return listGlobalReviews(10000, undefined as any); // Fetch all reviews
      } else if (reviewFilter === 'active') {
        return listGlobalReviews(10000, true);
      } else if (reviewFilter === 'in_active') {
        return listGlobalReviews(10000, false);
      }
    },
  });

  // Form setups
  const detailsForm = useForm<UpdateUserFormValues>({ resolver: zodResolver(updateUserFormSchema) });
  const passwordForm = useForm<UpdatePasswordFormValues>({ resolver: zodResolver(updatePasswordFormSchema) });
  const reviewForm = useForm<CreateReviewFormValues>({
    resolver: zodResolver(createReviewFormSchema),
    defaultValues: { post_id: null, is_active: true }, // Default is_active to true
  });
  const editReviewForm = useForm<UpdateReviewFormValues>({
    resolver: zodResolver(updateReviewFormSchema),
  });

  // Mutations
  const updateUserMutation = useMutation({
    mutationFn: updateUser,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['user'] });
      setIsDetailsModalOpen(false);
      detailsForm.reset();
    },
    onError: (err) => console.error('Update user error:', err),
  });

  const updatePasswordMutation = useMutation({
    mutationFn: updateUserPassword,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['user'] });
      setIsPasswordModalOpen(false);
      passwordForm.reset();
    },
    onError: (err) => console.error('Update password error:', err),
  });

  const createReviewMutation = useMutation({
    mutationFn: createReview,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['globalReviews'] });
      setIsReviewModalOpen(false);
      reviewForm.reset({ post_id: null, is_active: true });
    },
    onError: (err) => console.error('Create review error:', err),
  });

  const updateReviewMutation = useMutation({
    mutationFn: updateReview,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['globalReviews'] });
      setIsEditReviewModalOpen(false);
      setSelectedReview(null);
    },
    onError: (err) => console.error('Update review error:', err),
  });

  const deleteReviewsMutation = useMutation({
    mutationFn: deleteManyReviews,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['globalReviews'] });
      setSelectedReviews([]);
    },
    onError: (err) => console.error('Delete reviews error:', err),
  });

  // Handlers
  const handleFileUpload = async (file: File, field: 'logo' | 'doctor_pic') => {
    try {
      setIsImageLoading((prev) => ({ ...prev, [field]: true }));
      const existingFiles = user?.[field];
      if (existingFiles && existingFiles.length > 0) {
        const oldFileUrl = existingFiles[0].url;
        removeFile(oldFileUrl);
      }
      const uploadResponse = await uploadFile(file);
      const fileMetadata = uploadResponse.data.files as FileMetadata[];
      const updateData = { [field]: fileMetadata };
      await updateUserMutation.mutateAsync(updateData);
    } catch (err) {
      console.error('File upload error:', err);
    } finally {
      setIsImageLoading((prev) => ({ ...prev, [field]: false }));
    }
  };

  const onDetailsSubmit = (data: UpdateUserFormValues) => updateUserMutation.mutate(data);
  const onPasswordSubmit = (data: UpdatePasswordFormValues) => updatePasswordMutation.mutate(data);
  const onReviewSubmit = (data: CreateReviewFormValues) => createReviewMutation.mutate(data);
  const onEditReviewSubmit = (data: UpdateReviewFormValues) => {
    if (selectedReview) updateReviewMutation.mutate({ id: selectedReview, ...data });
  };

  const handleDeleteReviews = () => {
    if (selectedReviews.length > 0) deleteReviewsMutation.mutate(selectedReviews);
  };

  const toggleReviewSelection = (id: number) => {
    setSelectedReviews((prev) => (prev.includes(id) ? prev.filter((r) => r !== id) : [...prev, id]));
  };

  const openEditReviewModal = (review: Review) => {
    setSelectedReview(review.id);
    editReviewForm.reset({ name: review.name, text: review.text, rate: review.rate, is_active: review.is_active });
    setIsEditReviewModalOpen(true);
  };

  return (
    <div className="px-2 py-4 min-h-screen" dir="rtl">
      <h2 className="text-xl sm:text-2xl font-bold mb-6 text-doctor-dark font-heading text-start">إدارة الملف الشخصي</h2>

      {isLoading ? (
        <div className="text-center py-4 text-gray-600">جارٍ التحميل...</div>
      ) : error ? (
        <p className="text-red-500 mb-4 text-center">{handleBackendError(error)}</p>
      ) : (
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Right Column: User Details, Password, and Reviews */}
          <div className="lg:col-span-2 space-y-6">
            {/* User Details Section */}
            <div className="border border-gray-200 rounded-lg p-6 bg-white transition-all duration-300">
              <h3 className="text-lg font-semibold text-doctor-primary font-heading mb-4 flex items-center gap-2">
                <FaUser className="text-doctor-primary" /> بيانات المستخدم
              </h3>
              <div className="space-y-4">
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">اسم المستخدم</label>
                  <input
                    type="text"
                    value={user?.user_name || ''}
                    className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                    placeholder="اسم المستخدم"
                    disabled
                  />
                </div>
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">البريد الإلكتروني</label>
                  <input
                    type="email"
                    value={user?.email || ''}
                    className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                    placeholder="البريد الإلكتروني"
                    disabled
                  />
                </div>
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">عن الطبيب</label>
                  <textarea
                    value={user?.about_dr || ''}
                    className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-300"
                    placeholder="عن الطبيب"
                    rows={4}
                    disabled
                  />
                </div>
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">مفتاح الدخول</label>
                  <input
                    type="text"
                    value={user?.login_key || ''}
                    className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                    placeholder="مفتاح الدخول"
                    disabled
                  />
                </div>
                <div className="flex justify-end">
                  <button
                    onClick={() => setIsDetailsModalOpen(true)}
                    className="px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary transition-colors duration-200"
                  >
                    تعديل البيانات
                  </button>
                </div>
              </div>
            </div>

            {/* User Password Section */}
            <div className="border border-gray-200 rounded-lg p-6 bg-white transition-all duration-300">
              <h3 className="text-lg font-semibold text-doctor-primary font-heading mb-4 flex items-center gap-2">
                <FaLock className="text-doctor-primary" /> كلمة المرور
              </h3>
              <div className="space-y-4">
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">كلمة المرور القديمة</label>
                  <input
                    type="password"
                    className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                    placeholder="كلمة المرور القديمة"
                    disabled
                  />
                </div>
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">كلمة المرور الجديدة</label>
                  <input
                    type="password"
                    className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                    placeholder="كلمة المرور الجديدة"
                    disabled
                  />
                </div>
                <div className="flex justify-end">
                  <button
                    onClick={() => setIsPasswordModalOpen(true)}
                    className="px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary transition-colors duration-200"
                  >
                    تغيير كلمة المرور
                  </button>
                </div>
              </div>
            </div>

            {/* Global Reviews Section */}
            <div className="border border-gray-200 rounded-lg p-6 bg-white transition-all duration-300">
              <div className="flex justify-between items-center mb-4">
                <h3 className="text-lg font-semibold text-doctor-primary font-heading flex items-center gap-2">
                  <FaStar className="text-doctor-primary" /> التقييمات العامة
                </h3>
                <div className="flex gap-2">
                  <select
                    value={reviewFilter}
                    onChange={(e) => setReviewFilter(e.target.value as 'all' | 'active' | 'in_active')}
                    className="px-2 py-1 border rounded-lg text-sm text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark"
                  >
                    <option value="active">نشط</option>
                    <option value="in_active">غير نشط</option>
                    <option value="all">الكل</option>
                  </select>
                  <button
                    onClick={() => setIsReviewModalOpen(true)}
                    className="px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary transition-colors duration-200"
                  >
                    إضافة تقييم
                  </button>
                </div>
              </div>
              {reviewsLoading ? (
                <div className="text-center py-4 text-gray-600">جارٍ التحميل...</div>
              ) : reviewsError ? (
                <p className="text-red-500 text-center">{handleBackendError(reviewsError)}</p>
              ) : !reviews || reviews.length === 0 ? (
                <p className="text-gray-500 text-center">لا توجد تقييمات عامة بعد</p>
              ) : (
                <>
                  <div className="space-y-4">
                    {reviews.map((review) => (
                      <div key={review.id} className="flex items-start gap-4 p-4 border border-neutral-200 rounded-lg">
                        <input type="checkbox" checked={selectedReviews.includes(review.id)} onChange={() => toggleReviewSelection(review.id)} className="mt-1" />
                        <div className="flex-1">
                          <div className="flex items-center gap-2">
                            <p className="font-semibold text-doctor-dark">{review.name}</p>
                            <div className="flex">
                              {[...Array(5)].map((_, i) => (
                                <FaStar key={i} className={i < review.rate ? 'text-yellow-400' : 'text-gray-300'} />
                              ))}
                            </div>
                            <span className={`text-sm px-2 py-1 rounded-full ${review.is_active ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'}`}>
                              {review.is_active ? 'نشط' : 'غير نشط'}
                            </span>
                          </div>
                          <p className="text-gray-600">{review.text}</p>
                          <p className="text-xs text-gray-500">{new Date(review.createdAt).toLocaleDateString('ar')}</p>
                          <button
                            onClick={() => openEditReviewModal(review)}
                            className="mt-2 bg-doctor-dark text-white text-sm font-medium px-3 py-1.5 rounded-md hover:bg-doctor-dark/90 focus:outline-none focus:ring-2 focus:ring-doctor-dark/50 transition duration-200 ease-in-out"
                          >
                            تعديل
                          </button>
                        </div>
                      </div>
                    ))}
                  </div>
                  {selectedReviews.length > 0 && (
                    <button
                      onClick={handleDeleteReviews}
                      disabled={deleteReviewsMutation.isPending}
                      className="mt-4 px-4 py-2 bg-red-500 text-white rounded-lg hover:bg-red-600 disabled:opacity-50 transition-colors duration-200 flex items-center gap-2"
                    >
                      <FaTrash /> حذف المحدد ({selectedReviews.length})
                    </button>
                  )}
                </>
              )}
            </div>
          </div>

          {/* Left Column: Logo and Doctor Picture */}
          <div className="space-y-6">
            {/* Logo Section */}
            <div className="border border-gray-200 rounded-lg p-6 bg-white transition-all duration-300">
              <h3 className="text-lg font-semibold text-doctor-primary font-heading mb-4 flex items-center gap-2">
                <FaImage className="text-doctor-primary" /> الشعار
              </h3>
              <div className="flex justify-center">
                {isImageLoading.logo ? (
                  <div className="w-32 h-32 flex items-center justify-center">
                    <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-doctor-primary"></div>
                  </div>
                ) : user?.logo && user.logo.length > 0 ? (
                  <Image width={1000} height={1000} src={user.logo[0].url} alt="Logo" className="w-32 h-32 object-cover rounded-lg" />
                ) : (
                  <div className="w-32 h-32 bg-gray-100 rounded-lg flex items-center justify-center text-gray-500">
                    <FaImage className="text-3xl" />
                  </div>
                )}
              </div>
              <div className="mt-4 flex justify-center">
                <label className="px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary transition-colors duration-200 cursor-pointer">
                  تحميل شعار جديد
                  <input
                    type="file"
                    accept="image/*"
                    className="hidden"
                    onChange={(e) => {
                      const file = e.target.files?.[0];
                      if (file) handleFileUpload(file, 'logo');
                    }}
                  />
                </label>
              </div>
            </div>

            {/* Doctor Picture Section */}
            <div className="border border-gray-200 rounded-lg p-6 bg-white transition-all duration-300">
              <h3 className="text-lg font-semibold text-doctor-primary font-heading mb-4 flex items-center gap-2">
                <FaImage className="text-doctor-primary" /> صورة الطبيب
              </h3>
              <div className="flex justify-center">
                {isImageLoading.doctor_pic ? (
                  <div className="w-32 h-32 flex items-center justify-center">
                    <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-doctor-primary"></div>
                  </div>
                ) : user?.doctor_pic && user.doctor_pic.length > 0 ? (
                  <Image width={1000} height={1000} src={user.doctor_pic[0].url} alt="Doctor Picture" className="w-32 h-32 object-cover rounded-lg" />
                ) : (
                  <div className="w-32 h-32 bg-gray-100 rounded-lg flex items-center justify-center text-gray-500">
                    <FaImage className="text-3xl" />
                  </div>
                )}
              </div>
              <div className="mt-4 flex justify-center">
                <label className="px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary transition-colors duration-200 cursor-pointer">
                  تحميل صورة جديدة
                  <input
                    type="file"
                    accept="image/*"
                    className="hidden"
                    onChange={(e) => {
                      const file = e.target.files?.[0];
                      if (file) handleFileUpload(file, 'doctor_pic');
                    }}
                  />
                </label>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal for Updating User Details */}
      {isDetailsModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsDetailsModalOpen(false)} />
          <div className="relative bg-white p-6 rounded-lg w-full max-w-md shadow-lg">
            <h3 className="text-xl font-bold mb-4 text-doctor-dark font-heading">تعديل البيانات</h3>
            <form onSubmit={detailsForm.handleSubmit(onDetailsSubmit)} className="space-y-4" dir="rtl">
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">اسم المستخدم</label>
                <input
                  {...detailsForm.register('user_name')}
                  defaultValue={user?.user_name || ''}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                  placeholder="اسم المستخدم"
                />
                {detailsForm.formState.errors.user_name && <p className="text-red-500 text-sm mt-1">{detailsForm.formState.errors.user_name.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">البريد الإلكتروني</label>
                <input
                  {...detailsForm.register('email')}
                  defaultValue={user?.email || ''}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                  placeholder="البريد الإلكتروني"
                />
                {detailsForm.formState.errors.email && <p className="text-red-500 text-sm mt-1">{detailsForm.formState.errors.email.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">عن الطبيب</label>
                <textarea
                  {...detailsForm.register('about_dr')}
                  defaultValue={user?.about_dr || ''}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                  placeholder="عن الطبيب"
                  rows={4}
                />
                {detailsForm.formState.errors.about_dr && <p className="text-red-500 text-sm mt-1">{detailsForm.formState.errors.about_dr.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">مفتاح الدخول</label>
                <input
                  {...detailsForm.register('login_key')}
                  defaultValue={user?.login_key || ''}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                  placeholder="مفتاح الدخول"
                />
                {detailsForm.formState.errors.login_key && <p className="text-red-500 text-sm mt-1">{detailsForm.formState.errors.login_key.message}</p>}
              </div>
              <div className="flex gap-2">
                <button
                  type="submit"
                  disabled={updateUserMutation.isPending}
                  className="flex-1 px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary disabled:opacity-50 transition-colors duration-200"
                >
                  {updateUserMutation.isPending ? 'جارٍ...' : 'حفظ'}
                </button>
                <button
                  type="button"
                  onClick={() => setIsDetailsModalOpen(false)}
                  className="flex-1 px-4 py-2 bg-gray-300 rounded-lg hover:bg-gray-400 transition-colors duration-200"
                >
                  إلغاء
                </button>
              </div>
              {updateUserMutation.isError && <p className="text-red-500 text-sm mt-2">{handleBackendError(updateUserMutation.error)}</p>}
            </form>
          </div>
        </div>
      )}

      {/* Modal for Updating Password */}
      {isPasswordModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsPasswordModalOpen(false)} />
          <div className="relative bg-white p-6 rounded-lg w-full max-w-md shadow-lg">
            <h3 className="text-xl font-bold mb-4 text-doctor-dark font-heading">تغيير كلمة المرور</h3>
            <form onSubmit={passwordForm.handleSubmit(onPasswordSubmit)} className="space-y-4" dir="rtl">
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">كلمة المرور القديمة</label>
                <input
                  type="password"
                  {...passwordForm.register('oldPassword')}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                  placeholder="كلمة المرور القديمة"
                />
                {passwordForm.formState.errors.oldPassword && <p className="text-red-500 text-sm mt-1">{passwordForm.formState.errors.oldPassword.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">كلمة المرور الجديدة</label>
                <input
                  type="password"
                  {...passwordForm.register('newPassword')}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                  placeholder="كلمة المرور الجديدة"
                />
                {passwordForm.formState.errors.newPassword && <p className="text-red-500 text-sm mt-1">{passwordForm.formState.errors.newPassword.message}</p>}
              </div>
              <div className="flex gap-2">
                <button
                  type="submit"
                  disabled={updatePasswordMutation.isPending}
                  className="flex-1 px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary disabled:opacity-50 transition-colors duration-200"
                >
                  {updatePasswordMutation.isPending ? 'جارٍ...' : 'حفظ'}
                </button>
                <button
                  type="button"
                  onClick={() => setIsPasswordModalOpen(false)}
                  className="flex-1 px-4 py-2 bg-gray-300 rounded-lg hover:bg-gray-400 transition-colors duration-200"
                >
                  إلغاء
                </button>
              </div>
              {updatePasswordMutation.isError && <p className="text-red-500 text-sm mt-2">{handleBackendError(updatePasswordMutation.error)}</p>}
            </form>
          </div>
        </div>
      )}

      {/* Modal for Creating Review */}
      {isReviewModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsReviewModalOpen(false)} />
          <div className="relative bg-white p-6 rounded-lg w-full max-w-md shadow-lg">
            <h3 className="text-xl font-bold mb-4 text-doctor-dark font-heading">إضافة تقييم عام</h3>
            <form onSubmit={reviewForm.handleSubmit(onReviewSubmit)} className="space-y-4" dir="rtl">
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الاسم</label>
                <input
                  {...reviewForm.register('name')}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                  placeholder="الاسم"
                />
                {reviewForm.formState.errors.name && <p className="text-red-500 text-sm mt-1">{reviewForm.formState.errors.name.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">النص</label>
                <textarea
                  {...reviewForm.register('text')}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                  placeholder="النص"
                  rows={4}
                />
                {reviewForm.formState.errors.text && <p className="text-red-500 text-sm mt-1">{reviewForm.formState.errors.text.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">التقييم (1-5)</label>
                <input
                  type="number"
                  {...reviewForm.register('rate', { valueAsNumber: true })}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                  placeholder="التقييم"
                  min={1}
                  max={5}
                />
                {reviewForm.formState.errors.rate && <p className="text-red-500 text-sm mt-1">{reviewForm.formState.errors.rate.message}</p>}
              </div>
              <div>
                <label className="flex items-center gap-2 text-sm font-medium text-gray-700">
                  <input type="checkbox" {...reviewForm.register('is_active')} className="rounded border-gray-300 text-doctor-dark focus:ring-doctor-dark" />
                  نشط
                </label>
                {reviewForm.formState.errors.is_active && <p className="text-red-500 text-sm mt-1">{reviewForm.formState.errors.is_active.message}</p>}
              </div>
              <div className="flex gap-2">
                <button
                  type="submit"
                  disabled={createReviewMutation.isPending}
                  className="flex-1 px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary disabled:opacity-50 transition-colors duration-200"
                >
                  {createReviewMutation.isPending ? 'جارٍ...' : 'حفظ'}
                </button>
                <button
                  type="button"
                  onClick={() => setIsReviewModalOpen(false)}
                  className="flex-1 px-4 py-2 bg-gray-300 rounded-lg hover:bg-gray-400 transition-colors duration-200"
                >
                  إلغاء
                </button>
              </div>
              {createReviewMutation.isError && <p className="text-red-500 text-sm mt-2">{handleBackendError(createReviewMutation.error)}</p>}
            </form>
          </div>
        </div>
      )}

      {/* Modal for Editing Review */}
      {isEditReviewModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsEditReviewModalOpen(false)} />
          <div className="relative bg-white p-6 rounded-lg w-full max-w-md shadow-lg">
            <h3 className="text-xl font-bold mb-4 text-doctor-dark font-heading">تعديل التقييم</h3>
            <form onSubmit={editReviewForm.handleSubmit(onEditReviewSubmit)} className="space-y-4" dir="rtl">
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الاسم</label>
                <input
                  {...editReviewForm.register('name')}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                  placeholder="الاسم"
                />
                {editReviewForm.formState.errors.name && <p className="text-red-500 text-sm mt-1">{editReviewForm.formState.errors.name.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">النص</label>
                <textarea
                  {...editReviewForm.register('text')}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                  placeholder="النص"
                  rows={4}
                />
                {editReviewForm.formState.errors.text && <p className="text-red-500 text-sm mt-1">{editReviewForm.formState.errors.text.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">التقييم (1-5)</label>
                <input
                  type="number"
                  {...editReviewForm.register('rate', { valueAsNumber: true })}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200"
                  placeholder="التقييم"
                  min={1}
                  max={5}
                />
                {editReviewForm.formState.errors.rate && <p className="text-red-500 text-sm mt-1">{editReviewForm.formState.errors.rate.message}</p>}
              </div>
              <div>
                <label className="flex items-center gap-2 text-sm font-medium text-gray-700">
                  <input type="checkbox" {...editReviewForm.register('is_active')} className="rounded border-gray-300 text-doctor-dark focus:ring-doctor-dark" />
                  نشط
                </label>
                {editReviewForm.formState.errors.is_active && <p className="text-red-500 text-sm mt-1">{editReviewForm.formState.errors.is_active.message}</p>}
              </div>
              <div className="flex gap-2">
                <button
                  type="submit"
                  disabled={updateReviewMutation.isPending}
                  className="flex-1 px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary disabled:opacity-50 transition-colors duration-200"
                >
                  {updateReviewMutation.isPending ? 'جارٍ...' : 'حفظ'}
                </button>
                <button
                  type="button"
                  onClick={() => setIsEditReviewModalOpen(false)}
                  className="flex-1 px-4 py-2 bg-gray-300 rounded-lg hover:bg-gray-400 transition-colors duration-200"
                >
                  إلغاء
                </button>
              </div>
              {updateReviewMutation.isError && <p className="text-red-500 text-sm mt-2">{handleBackendError(updateReviewMutation.error)}</p>}
            </form>
          </div>
        </div>
      )}
    </div>
  );
};

export default UserProfile;
