import Cookies from 'js-cookie';
import { Review, CreateReviewFormValues, UpdateReviewFormValues, ApiResponse } from './reviews.schema';

const API_URL = process.env.NEXT_PUBLIC_SERVER_URL || 'http://localhost:7000';

const getAuthToken = () => {
  if (typeof window !== 'undefined') {
    return Cookies.get('dr_auth_token');
  }
  return null;
};

// List global reviews (post_id = null)
export const listGlobalReviews = async (limit?: number, isActive?: boolean): Promise<Review[]> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const url = new URL(`${API_URL}/review/list`);
  if (limit) url.searchParams.append('limit', limit.toString());
  if (isActive !== undefined) url.searchParams.append('is_active', isActive.toString()); // Filter by is_active

  const response = await fetch(url, {
    method: 'GET',
    headers: {
      Authorization: `Bearer ${token}`,
    },
  });

  const result = await response.json();
  if (!response.ok) throw result;

  return result as Review[];
};

// List post-specific reviews
export const listPostReviews = async (selectedPostId: number, isActive?: boolean) => {
  const url = new URL(`${API_URL}/review/list`);
  url.searchParams.append('post_id', selectedPostId!.toString());
  if (isActive !== undefined) url.searchParams.append('is_active', isActive.toString()); // Filter by is_active
  // Filter by is_active

  const response = await fetch(url, {
    method: 'GET',
  });

  const result = await response.json();
  if (!response.ok) throw result;
  return result as Review[];
};

// Create a review (authenticated, post-related)
export const createReview = async (data: CreateReviewFormValues): Promise<ApiResponse> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const reviewData = { ...data, is_active: data.is_active ?? true }; // Default is_active to true

  const response = await fetch(`${API_URL}/review/create`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(reviewData),
  });

  const result = await response.json();
  if (!response.ok) throw result;
  return result as ApiResponse;
};

// Create a public review (no auth, global)
export const createPublicReview = async (data: CreateReviewFormValues): Promise<ApiResponse> => {
  const reviewData = { ...data, is_active: data.is_active ?? false }; // Default is_active to false

  const response = await fetch(`${API_URL}/review/create-from-site`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify(reviewData),
  });

  const result = await response.json();
  if (!response.ok) throw result;
  return result as ApiResponse;
};

// Update a review
export const updateReview = async (data: UpdateReviewFormValues & { id: number }): Promise<ApiResponse> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const { id, ...restBody } = data;

  const response = await fetch(`${API_URL}/review/${id}`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(restBody),
  });

  const result = await response.json();
  if (!response.ok) throw result;
  return result as ApiResponse;
};

// Delete multiple reviews
export const deleteManyReviews = async (ids: number[]): Promise<ApiResponse> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/review/delete-many`, {
    method: 'DELETE',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify({ ids }),
  });

  const result = await response.json();
  if (!response.ok) throw result;
  return result as ApiResponse;
};

// Get review details
export const getReviewDetails = async (id: number): Promise<ApiResponse> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/review/${id}`, {
    method: 'GET',
    headers: {
      Authorization: `Bearer ${token}`,
    },
  });

  const result = await response.json();
  if (!response.ok) throw result;
  return result as ApiResponse;
};
