import { z } from 'zod';

export interface Review {
  id: number;
  name: string;
  text: string;
  rate: number;
  post_id: number | null;
  is_active: boolean; // New field
  createdAt: string;
  updatedAt: string;
}

export const createReviewFormSchema = z.object({
  name: z.string().min(1, 'الاسم مطلوب').max(256, 'الاسم يجب ألا يتجاوز 256 حرفًا'),
  text: z.string().min(1, 'النص مطلوب').max(1000, 'النص يجب ألا يتجاوز 1000 حرف'),
  rate: z.number().int().min(1, 'التقييم يجب أن يكون 1 على الأقل').max(5, 'التقييم يجب ألا يتجاوز 5'),
  post_id: z.number().int().positive().nullable(), // Allow null or a positive integer
  is_active: z.boolean().optional().default(true), // Optional, defaults to true
});

export const updateReviewFormSchema = z.object({
  name: z.string().min(1, 'الاسم مطلوب').max(256, 'الاسم يجب ألا يتجاوز 256 حرفًا').optional(),
  text: z.string().min(1, 'النص مطلوب').max(1000, 'النص يجب ألا يتجاوز 1000 حرف').optional(),
  rate: z.number().int().min(1, 'التقييم يجب أن يكون 1 على الأقل').max(5, 'التقييم يجب ألا يتجاوز 5').optional(),
  post_id: z.number().int().positive().nullable().optional(), // Allow updates to post_id if needed
  is_active: z.boolean().optional(), // Allow updating is_active
});

export const apiResponseSchema = z.object({
  statusCode: z.number(),
  data: z.array(
    z.object({
      id: z.number(),
      name: z.string(),
      text: z.string(),
      rate: z.number(),
      post_id: z.number().nullable(),
      is_active: z.boolean(), // New field
      createdAt: z.string(),
      updatedAt: z.string(),
    }),
  ),
});

export type CreateReviewFormValues = z.infer<typeof createReviewFormSchema>;
export type UpdateReviewFormValues = z.infer<typeof updateReviewFormSchema>;
export type ApiResponse = z.infer<typeof apiResponseSchema>;
