import Cookies from 'js-cookie';
import { CreatePostFormValues, UpdatePostFormValues, Post, ListPostsFilter, FileMetadata } from './post.schema';

const API_URL = process.env.NEXT_PUBLIC_SERVER_URL || 'http://localhost:3000';

const getAuthToken = () => {
  if (typeof window !== 'undefined') {
    return Cookies.get('dr_auth_token');
  }
  return null;
};

export const listPosts = async (filter: ListPostsFilter): Promise<{ posts: Post[]; total: number }> => {
  const queryParams = new URLSearchParams({
    page: filter.page.toString(),
    pageSize: filter.pageSize.toString(),
    ...(filter.q && { q: filter.q }),
    ...(filter.category_id && { category_id: filter.category_id.toString() }),
    ...(filter.order && { order: JSON.stringify(filter.order) }),
  });

  const response = await fetch(`${API_URL}/post/list?${queryParams.toString()}`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }
  return result as { posts: Post[]; total: number };
};

export const createPost = async (data: CreatePostFormValues): Promise<Post> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/post/create`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }
  return result as Post;
};

export const getPostDetails = async (id: number): Promise<Post> => {
  const response = await fetch(`${API_URL}/post/${id}`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }
  return result as Post;
};

export const updatePost = async (id: number, data: UpdatePostFormValues): Promise<Post> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/post/${id}`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }
  return result as Post;
};

export const deletePosts = async (ids: number[]): Promise<void> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/post/delete-many`, {
    method: 'DELETE',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify({ ids }),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }
  return result;
};

export const uploadFile = async (file: File): Promise<{ message: string; data: { files: FileMetadata[] } }> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const formData = new FormData();
  formData.append('files', file);

  const response = await fetch(`${API_URL}/multerUploader/upload`, {
    method: 'POST',
    headers: {
      Authorization: `Bearer ${token}`,
    },
    body: formData,
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }
  return result as { message: string; data: { files: FileMetadata[] } };
};

// Remove file (delete old image from server)
export const removeFile = async (fileUrl: string): Promise<void> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  await fetch(`${API_URL}/multerUploader/remove`, {
    method: 'DELETE',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify({
      files: [{ url: fileUrl }],
    }),
  });
};
