import { z } from 'zod';

export interface FileMetadata {
  url: string;
  filename: string;
}

export const contentBlockSchema = z.discriminatedUnion('type', [
  z.object({ type: z.literal('paragraph'), value: z.string().min(1, 'الفقرة مطلوبة') }),
  z.object({ type: z.literal('subtitle'), value: z.string().min(1, 'العنوان الفرعي مطلوب') }),
  z.object({
    type: z.literal('unordered-list'),
    value: z.array(z.string().min(1, 'العنصر مطلوب')).min(1, 'القائمة يجب أن تحتوي على عنصر واحد على الأقل'),
  }),
  z.object({
    type: z.literal('ordered-list'),
    value: z.array(z.string().min(1, 'العنصر مطلوب')).min(1, 'القائمة يجب أن تحتوي على عنصر واحد على الأقل'),
  }),
  z.object({ type: z.literal('image'), value: z.string().url('يجب أن يكون رابط صورة صالح') }),
  z.object({ type: z.literal('youtube'), value: z.string().url('يجب أن يكون رابط YouTube صالح') }),
]);

export interface Post {
  id: number;
  title: string;
  description: string;
  content: { type: string; value: string | string[] }[];
  thumbnail: FileMetadata[] | null;
  category_id?: number;
  category_name?: string;
  createdAt: string;
  updatedAt: string;
}

export const createPostFormSchema = z.object({
  title: z.string().min(1, 'العنوان مطلوب').max(256, 'العنوان يجب ألا يتجاوز 256 حرفًا'),
  description: z.string().max(1000, 'الوصف يجب ألا يتجاوز 1000 حرف').optional().default(''),
  content: z.array(contentBlockSchema).min(1, 'يجب إضافة محتوى واحد على الأقل'),
  thumbnail: z
    .array(z.object({ url: z.string(), filename: z.string() }))
    .optional()
    .default([]),
  category_id: z
    .number({ invalid_type_error: 'الفئة مطلوبة' })
    .int({ message: 'الفئة يجب أن تكون عددًا صحيحًا' })
    .positive({ message: 'الفئة يجب أن تكون قيمة موجبة' })
    .refine((val) => !isNaN(val), { message: 'الفئة مطلوبة' })
    .optional(),
});

export const updatePostFormSchema = createPostFormSchema;

export const listPostsFilterSchema = z.object({
  page: z.number().min(1).optional().default(1),
  pageSize: z.number().min(1).optional().default(10),
  q: z.string().max(200).optional(),
  category_id: z.number().int().positive().optional(),
  order: z.array(z.object({ column: z.enum(['id', 'title', 'description', 'category_id']), direction: z.enum(['asc', 'desc']) })).optional(),
});

export type CreatePostFormValues = z.infer<typeof createPostFormSchema>;
export type UpdatePostFormValues = z.infer<typeof updatePostFormSchema>;
export type ListPostsFilter = z.infer<typeof listPostsFilterSchema>;
