/* eslint-disable @typescript-eslint/no-explicit-any */
/* eslint-disable @typescript-eslint/no-unused-vars */
'use client';

import { closestCenter, DndContext, PointerSensor, useSensor, useSensors } from '@dnd-kit/core';
import { SortableContext, useSortable, verticalListSortingStrategy } from '@dnd-kit/sortable';
import { CSS } from '@dnd-kit/utilities';
import { zodResolver } from '@hookform/resolvers/zod';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import Image from 'next/image';
import Link from 'next/link';
import { useState } from 'react';
import { useFieldArray, useForm } from 'react-hook-form';
import { FaGripVertical, FaHeading, FaImage, FaListOl, FaListUl, FaParagraph, FaPlus, FaStar, FaTrash, FaYoutube } from 'react-icons/fa';
import { MdEdit } from 'react-icons/md';
import { listCategories } from '../categories/category.service';
import { createReviewFormSchema, CreateReviewFormValues, updateReviewFormSchema, UpdateReviewFormValues } from '../reviews/reviews.schema';
import { createReview, deleteManyReviews, listPostReviews, updateReview } from '../reviews/reviews.service';
import { createPostFormSchema, CreatePostFormValues, updatePostFormSchema, UpdatePostFormValues } from './post.schema';
import { createPost, deletePosts, getPostDetails, listPosts, updatePost, uploadFile, removeFile } from './post.service';

const Posts = () => {
  const queryClient = useQueryClient();
  const [page, setPage] = useState(1);
  const [categoryId, setCategoryId] = useState<number | undefined>(undefined);
  const [selectedIds, setSelectedIds] = useState<number[]>([]);
  const [isCreateModalOpen, setIsCreateModalOpen] = useState(false);
  const [isEditModalOpen, setIsEditModalOpen] = useState(false);
  const [isReviewModalOpen, setIsReviewModalOpen] = useState(false);
  const [isEditReviewModalOpen, setIsEditReviewModalOpen] = useState(false);
  const [isViewReviewsModalOpen, setIsViewReviewsModalOpen] = useState(false);
  const [editingPostId, setEditingPostId] = useState<number | null>(null);
  const [selectedPostId, setSelectedPostId] = useState<number | null>(null);
  const [selectedReviewId, setSelectedReviewId] = useState<number | null>(null);
  const [selectedReviewIds, setSelectedReviewIds] = useState<number[]>([]);
  const [thumbnailPreview, setThumbnailPreview] = useState<string | null>(null);
  const [uploadError, setUploadError] = useState<string | null>(null);
  const [isThumbnailLoading, setIsThumbnailLoading] = useState(false);
  const [contentImageLoadingIndices, setContentImageLoadingIndices] = useState<number[]>([]);
  const [reviewFilter, setReviewFilter] = useState<'all' | 'active' | 'in_active'>('all'); // New filter state
  const pageSize = 10;

  const { data, isLoading, error } = useQuery({
    queryKey: ['posts', page, categoryId],
    queryFn: () => listPosts({ page, pageSize, category_id: categoryId }),
  });

  const postItems = data?.posts || [];
  const totalItems = data?.total || 0;
  const totalPages = Math.ceil(totalItems / pageSize);

  const { data: categories } = useQuery({
    queryKey: ['categories'],
    queryFn: () => listCategories(),
  });
  const subCategories = categories?.filter((subCat) => subCat.parent_category_id !== null && subCat.is_active);

  const { data: postReviews, isLoading: reviewsLoading } = useQuery({
    queryKey: ['postReviews', selectedPostId],
    queryFn: () => {
      if (reviewFilter === 'all') {
        return listPostReviews(selectedPostId as number, undefined as any);
      } else if (reviewFilter === 'active') {
        return listPostReviews(selectedPostId as number, true);
      } else if (reviewFilter === 'in_active') {
        return listPostReviews(selectedPostId as number, false);
      }
    }, // No isActive parameter
    enabled: !!selectedPostId && isViewReviewsModalOpen,
  });

  // Filter reviews client-side based on reviewFilter
  const filteredReviews =
    postReviews?.filter((review: any) => {
      if (reviewFilter === 'all') return true;
      return reviewFilter === 'active' ? review.is_active : !review.is_active;
    }) || [];

  const createForm = useForm<CreatePostFormValues>({
    resolver: zodResolver(createPostFormSchema),
    defaultValues: { title: '', description: '', content: [], thumbnail: [], category_id: undefined },
  });
  const {
    fields: createFields,
    append: createAppend,
    remove: createRemove,
    move: createMove,
  } = useFieldArray({
    control: createForm.control,
    name: 'content',
  });

  const updateForm = useForm<UpdatePostFormValues>({
    resolver: zodResolver(updatePostFormSchema),
    defaultValues: { title: '', description: '', content: [], thumbnail: [], category_id: undefined },
  });
  const {
    fields: updateFields,
    append: updateAppend,
    remove: updateRemove,
    move: updateMove,
  } = useFieldArray({
    control: updateForm.control,
    name: 'content',
  });

  const reviewForm = useForm<CreateReviewFormValues>({
    resolver: zodResolver(createReviewFormSchema),
    defaultValues: { name: '', text: '', rate: 1, post_id: null, is_active: true }, // Default is_active to true
  });

  const editReviewForm = useForm<UpdateReviewFormValues>({
    resolver: zodResolver(updateReviewFormSchema),
    defaultValues: { name: '', text: '', rate: 1, post_id: null, is_active: true },
  });

  const createMutation = useMutation({
    mutationFn: createPost,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['posts'] });
      setIsCreateModalOpen(false);
      setThumbnailPreview(null);
      setUploadError(null);
      setIsThumbnailLoading(false);
      createForm.reset();
    },
    onError: (err: any) => setUploadError('فشل في إنشاء المنشور: ' + (err.message || 'خطأ غير معروف')),
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: UpdatePostFormValues }) => updatePost(id, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['posts'] });
      setIsEditModalOpen(false);
      setEditingPostId(null);
      setThumbnailPreview(null);
      setUploadError(null);
      setIsThumbnailLoading(false);
      setContentImageLoadingIndices([]);
      updateForm.reset();
    },
    onError: (err: any) => setUploadError('فشل في تحديث المنشور: ' + (err.message || 'خطأ غير معروف')),
  });

  const deleteMutation = useMutation({
    mutationFn: deletePosts,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['posts'] });
      setSelectedIds([]);
    },
    onError: (err: any) => setUploadError('فشل في حذف المنشورات: ' + (err.message || 'خطأ غير معروف')),
  });

  const createReviewMutation = useMutation({
    mutationFn: createReview,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['postReviews', selectedPostId] });
      setIsReviewModalOpen(false);
      reviewForm.reset({ name: '', text: '', rate: 1, post_id: selectedPostId, is_active: true });
    },
    onError: (err: any) => setUploadError('فشل في إضافة التقييم: ' + (err.message || 'خطأ غير معروف')),
  });

  const updateReviewMutation = useMutation({
    mutationFn: (data: UpdateReviewFormValues & { id: number }) => updateReview(data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['postReviews', selectedPostId] });
      setIsEditReviewModalOpen(false);
      setSelectedReviewId(null);
      editReviewForm.reset();
    },
    onError: (err: any) => setUploadError('فشل في تحديث التقييم: ' + (err.message || 'خطأ غير معروف')),
  });

  const deleteReviewMutation = useMutation({
    mutationFn: deleteManyReviews,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['postReviews', selectedPostId] });
      setSelectedReviewIds([]);
    },
    onError: (err: any) => setUploadError('فشل في حذف التقييمات: ' + (err.message || 'خطأ غير معروف')),
  });

  const sensors = useSensors(useSensor(PointerSensor, { activationConstraint: { distance: 8 } }));

  const handlePageChange = (newPage: number) => {
    if (newPage >= 1 && newPage <= totalPages) {
      setPage(newPage);
      setSelectedIds([]);
    }
  };

  const handleCategoryFilter = (e: React.ChangeEvent<HTMLSelectElement>) => {
    const value = e.target.value ? Number(e.target.value) : undefined;
    setCategoryId(value);
    setPage(1);
    setSelectedIds([]);
  };

  const handleSelectPost = (id: number) => {
    setSelectedIds((prev) => (prev.includes(id) ? prev.filter((item) => item !== id) : [...prev, id]));
  };

  const handleCreateSubmit = (data: CreatePostFormValues) => createMutation.mutate(data);

  const handleEditClick = async (id: number, e: React.MouseEvent) => {
    e.preventDefault();
    try {
      const post = await getPostDetails(id);
      setEditingPostId(id);
      updateForm.reset({
        title: post.title,
        description: post.description || '',
        content: post.content as any,
        thumbnail: post.thumbnail || [],
        category_id: post.category_id,
      });
      setThumbnailPreview(post.thumbnail?.[0]?.url || null);
      setIsEditModalOpen(true);
      setUploadError(null);
      setIsThumbnailLoading(false);
      setContentImageLoadingIndices([]);
    } catch (err) {
      setUploadError('فشل في جلب تفاصيل المنشور');
    }
  };

  const handleUpdateSubmit = (data: UpdatePostFormValues) => {
    if (editingPostId) updateMutation.mutate({ id: editingPostId, data });
  };

  const handleThumbnailChange = async (e: React.ChangeEvent<HTMLInputElement>, form: any) => {
    const file = e.target.files?.[0];
    if (file) {
      try {
        setIsThumbnailLoading(true);
        setUploadError(null);
        const existingThumbnail = form.getValues('thumbnail');
        if (existingThumbnail && existingThumbnail.length > 0) {
          const oldFileUrl = existingThumbnail[0].url;
          removeFile(oldFileUrl);
        }
        const uploadResponse = await uploadFile(file);
        const thumbnailData = uploadResponse.data.files;
        if (thumbnailData.length === 0) throw new Error('No thumbnail data returned');
        form.setValue('thumbnail', thumbnailData);
        setThumbnailPreview(thumbnailData[0].url);
      } catch (err: any) {
        setUploadError('فشل في رفع الصورة: ' + (err.message || 'خطأ غير معروف'));
        form.setValue('thumbnail', []);
      } finally {
        setIsThumbnailLoading(false);
      }
    } else {
      setUploadError('يرجى اختيار صورة');
      form.setValue('thumbnail', []);
      setIsThumbnailLoading(false);
    }
  };

  const handleImageUpload = async (e: React.ChangeEvent<HTMLInputElement>, form: any, index: number) => {
    const file = e.target.files?.[0];
    if (file) {
      try {
        setContentImageLoadingIndices((prev) => [...prev, index]);
        setUploadError(null);
        const existingImage = form.getValues(`content.${index}.value`);
        if (existingImage && typeof existingImage === 'string' && existingImage.trim() !== '') {
          removeFile(existingImage);
        }
        const uploadResponse = await uploadFile(file);
        const imageData = uploadResponse.data.files;
        if (imageData.length === 0) throw new Error('No image data returned');
        form.setValue(`content.${index}.value`, imageData[0].url);
      } catch (err: any) {
        setUploadError('فشل في رفع الصورة: ' + (err.message || 'خطأ غير معروف'));
        form.setValue(`content.${index}.value`, '');
      } finally {
        setContentImageLoadingIndices((prev) => prev.filter((i) => i !== index));
      }
    } else {
      setUploadError('يرجى اختيار صورة');
      form.setValue(`content.${index}.value`, '');
      setContentImageLoadingIndices((prev) => prev.filter((i) => i !== index));
    }
  };

  const openCreateModal = () => {
    createForm.reset();
    setThumbnailPreview(null);
    setUploadError(null);
    setIsThumbnailLoading(false);
    setContentImageLoadingIndices([]);
    setIsCreateModalOpen(true);
  };

  const addContentBlock = (form: any, append: any, type: CreatePostFormValues['content'][0]['type']) => {
    switch (type) {
      case 'paragraph':
      case 'subtitle':
      case 'image':
      case 'youtube':
        append({ type, value: '' });
        break;
      case 'unordered-list':
      case 'ordered-list':
        append({ type, value: [''] });
        break;
    }
  };

  const handleDeleteSelected = () => deleteMutation.mutate(selectedIds);

  const handleAddReviewClick = (postId: number) => {
    setSelectedPostId(postId);
    reviewForm.reset({ name: '', text: '', rate: 1, post_id: postId, is_active: true });
    setIsReviewModalOpen(true);
    setUploadError(null);
  };

  const handleViewReviewsClick = (postId: number) => {
    setSelectedPostId(postId);
    setReviewFilter('all'); // Default to showing all reviews
    setIsViewReviewsModalOpen(true);
    setSelectedReviewIds([]);
    setUploadError(null);
  };

  const handleReviewSubmit = (data: CreateReviewFormValues) => createReviewMutation.mutate(data);

  const handleEditReviewClick = (review: any) => {
    setSelectedReviewId(review.id);
    editReviewForm.reset({ name: review.name, text: review.text, rate: review.rate, post_id: review.post_id, is_active: review.is_active });
    setIsEditReviewModalOpen(true);
    setUploadError(null);
  };

  const handleUpdateReviewSubmit = (data: UpdateReviewFormValues) => {
    if (selectedReviewId) updateReviewMutation.mutate({ id: selectedReviewId, ...data });
  };

  const handleDeleteReviews = () => {
    if (selectedReviewIds.length > 0) deleteReviewMutation.mutate(selectedReviewIds);
  };

  const toggleReviewSelection = (id: number) => {
    setSelectedReviewIds((prev) => (prev.includes(id) ? prev.filter((r) => r !== id) : [...prev, id]));
  };

  const ListItems = ({ form, index, field }: { form: any; index: number; field: { type: string; value: string[] } }) => {
    const { fields, append, remove } = useFieldArray({ control: form.control, name: `content.${index}.value` });
    return (
      <div className="space-y-2 p-5">
        {fields.map((item, i) => (
          <div key={item.id} className="flex gap-2 items-center">
            {field.type === 'unordered-list' ? <span className="text-gray-500">•</span> : <span className="text-gray-500">{i + 1}.</span>}
            <input
              {...form.register(`content.${index}.value.${i}`)}
              className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm"
              placeholder="عنصر القائمة..."
              defaultValue={(item as any).value}
            />
            <button
              type="button"
              onClick={() => remove(i)}
              className="w-8 h-8 flex items-center justify-center bg-red-100 text-red-600 rounded-full hover:bg-red-200 hover:text-red-700 transition-colors shadow-sm"
            >
              <FaTrash size={14} />
            </button>
          </div>
        ))}
        <button type="button" onClick={() => append('عنصر جديد')} className="text-blue-600 hover:text-blue-800 text-sm font-medium flex items-center gap-1">
          <FaPlus size={12} /> إضافة عنصر
        </button>
      </div>
    );
  };

  const contentTypeLabels: Record<string, string> = {
    paragraph: 'فقرة',
    subtitle: 'عنوان فرعي',
    'unordered-list': 'قائمة غير مرتبة',
    'ordered-list': 'قائمة مرتبة',
    image: 'صورة',
    youtube: 'فيديو YouTube',
  };

  const SortableItem = ({ id, field, index, form, remove }: { id: string; field: any; index: number; form: any; remove: (index: number) => void }) => {
    const { attributes, listeners, setNodeRef, transform, transition } = useSortable({ id });
    const style = { transform: CSS.Transform.toString(transform), transition };
    const isLoading = contentImageLoadingIndices.includes(index);

    return (
      <div ref={setNodeRef} style={style} className="mb-4 p-4 bg-gray-50 rounded-lg relative shadow-sm border border-gray-200">
        <div className="flex items-center justify-between mb-2">
          <div className="flex items-center gap-2">
            <button type="button" {...attributes} {...listeners} className="cursor-grab text-gray-600 hover:text-gray-800">
              <FaGripVertical size={16} />
            </button>
            <span className="inline-block bg-blue-100 text-blue-800 text-xs font-semibold px-2 py-1 rounded-full">{contentTypeLabels[field.type] || field.type}</span>
          </div>
          <button
            type="button"
            onClick={() => remove(index)}
            className="w-8 h-8 flex items-center justify-center bg-red-100 text-red-600 rounded-full hover:bg-red-200 hover:text-red-700 transition-colors shadow-sm"
          >
            <FaTrash size={16} />
          </button>
        </div>
        <div>
          {field.type === 'paragraph' || field.type === 'subtitle' ? (
            <textarea
              {...form.register(`content.${index}.value`)}
              className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm min-h-[80px]"
              placeholder={field.type === 'paragraph' ? 'اكتب فقرة...' : 'اكتب عنوان فرعي...'}
            />
          ) : field.type === 'image' ? (
            <div>
              <label className="w-full px-4 py-2 border border-gray-300 rounded-lg flex items-center gap-2 cursor-pointer text-sm bg-white shadow-sm hover:bg-gray-50 transition-colors">
                <FaImage className="text-gray-600" /> اختر صورة
                <input type="file" accept="image/*" className="hidden" onChange={(e) => handleImageUpload(e, form, index)} />
              </label>
              {isLoading ? (
                <div className="mt-2 w-24 h-24 flex items-center justify-center">
                  <div className="animate-spin rounded-full h-8 w-8 border-t-2 border-b-2 border-blue-500"></div>
                </div>
              ) : form.watch(`content.${index}.value`) ? (
                <div className="mt-2">
                  <Image
                    width={120}
                    height={120}
                    src={form.watch(`content.${index}.value`) as string}
                    alt="Content Image Preview"
                    className="w-24 h-24 object-cover rounded-lg shadow-sm"
                  />
                </div>
              ) : null}
            </div>
          ) : field.type === 'youtube' ? (
            <input
              {...form.register(`content.${index}.value`)}
              className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm"
              placeholder="رابط YouTube..."
            />
          ) : (
            <ListItems form={form} index={index} field={field} />
          )}
          {form.formState.errors.content?.[index]?.value && <p className="text-red-500 text-xs mt-1">{form.formState.errors.content[index]?.value?.message}</p>}
        </div>
      </div>
    );
  };

  const handleCreateDragEnd = (event: any) => {
    const { active, over } = event;
    if (active.id !== over.id) {
      const oldIndex = createFields.findIndex((field) => field.id === active.id);
      const newIndex = createFields.findIndex((field) => field.id === over.id);
      createMove(oldIndex, newIndex);
    }
  };

  const handleUpdateDragEnd = (event: any) => {
    const { active, over } = event;
    if (active.id !== over.id) {
      const oldIndex = updateFields.findIndex((field) => field.id === active.id);
      const newIndex = updateFields.findIndex((field) => field.id === over.id);
      updateMove(oldIndex, newIndex);
    }
  };

  return (
    <div className="px-4 py-6 min-h-screen" dir="rtl">
      <h2 className="text-2xl font-bold mb-6 text-doctor-dark font-heading text-start">إدارة المقالات</h2>

      {/* Filter and Actions */}
      <div className="flex flex-col sm:flex-row gap-4 mb-6 items-center justify-between">
        <div className="flex flex-wrap gap-3">
          <label className="flex items-center gap-2 text-sm text-gray-700">
            تصفية حسب الفئة:
            <select
              value={categoryId || ''}
              onChange={handleCategoryFilter}
              className="px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 bg-white shadow-sm"
            >
              <option value="">جميع الفئات</option>
              {subCategories?.map((cat) => (
                <option key={cat.id} value={cat.id}>
                  {cat.name}
                </option>
              ))}
            </select>
          </label>
        </div>
        <div className="flex flex-col sm:flex-row gap-3 w-full sm:w-auto">
          <button
            onClick={openCreateModal}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 flex items-center justify-center gap-2 w-full sm:w-auto shadow-md transition-colors"
          >
            <FaPlus /> إضافة منشور
          </button>
          {selectedIds.length > 0 && (
            <button
              onClick={handleDeleteSelected}
              className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 flex items-center justify-center gap-2 w-full sm:w-auto shadow-md transition-colors"
              disabled={deleteMutation.isPending}
            >
              <FaTrash /> حذف ({selectedIds.length})
            </button>
          )}
        </div>
      </div>

      {/* Posts List */}
      {isLoading ? (
        <div className="text-center py-12 text-gray-600 text-lg">جارٍ التحميل...</div>
      ) : error ? (
        <div className="text-center py-12 text-red-500 text-lg">حدث خطأ: {JSON.stringify(error)}</div>
      ) : postItems.length > 0 ? (
        <div className="grid gap-4 sm:grid-cols-1 md:grid-cols-2 lg:grid-cols-3">
          {postItems.map((post) => (
            <div
              key={post.id}
              className={`group bg-white rounded-xl shadow-md transition-all duration-300 border ${
                selectedIds.includes(post.id) ? 'border-blue-500 border-2' : 'border-gray-200'
              } p-4 flex flex-col`}
            >
              <Link href={`/posts/${post.id}`} className="flex-shrink-0">
                {post.thumbnail && post.thumbnail.length > 0 ? (
                  <div className="relative w-full h-40">
                    <Image src={post.thumbnail[0].url} alt={post.title} fill className="object-cover rounded-lg transition-transform duration-300 group-hover:scale-102" />
                  </div>
                ) : (
                  <div className="relative w-full h-40 bg-gray-100 rounded-lg flex items-center justify-center">
                    <span className="text-gray-400 text-sm">بدون صورة</span>
                  </div>
                )}
              </Link>
              <div className="flex-1 flex flex-col gap-2 mt-3">
                {post.category_name && <div className="inline-block w-fit rounded-full bg-cyan-100 px-3 py-1 text-xs font-semibold text-cyan-600">{post.category_name}</div>}
                <Link href={`/posts/${post.id}`}>
                  <h3 className="text-lg font-bold text-gray-800 font-heading line-clamp-2">{post.title}</h3>
                </Link>
                <p className="text-sm text-gray-600 font-body line-clamp-2">{post.description || 'بدون وصف'}</p>
              </div>
              <div className="flex items-center justify-between mt-4">
                <div className="flex gap-2">
                  <button
                    onClick={(e) => handleEditClick(post.id, e)}
                    className="p-2 text-blue-600 hover:text-blue-800 transition-colors rounded-full hover:bg-blue-100"
                    title="تعديل"
                  >
                    <MdEdit size={18} />
                  </button>
                  <button
                    onClick={() => handleAddReviewClick(post.id)}
                    className="p-2 text-green-600 hover:text-green-800 transition-colors rounded-full hover:bg-green-100"
                    title="إضافة تقييم"
                  >
                    <FaPlus size={18} />
                  </button>
                  <button
                    onClick={() => handleViewReviewsClick(post.id)}
                    className="p-2 text-yellow-600 hover:text-yellow-800 transition-colors rounded-full hover:bg-yellow-100"
                    title="عرض التقييمات"
                  >
                    <FaStar size={18} />
                  </button>
                </div>
                <input
                  type="checkbox"
                  className="h-5 w-5 text-blue-600 rounded focus:ring-blue-500"
                  checked={selectedIds.includes(post.id)}
                  onChange={() => handleSelectPost(post.id)}
                />
              </div>
            </div>
          ))}
        </div>
      ) : (
        <div className="text-center py-12 text-gray-600 text-lg">لا توجد منشورات</div>
      )}

      {/* Pagination */}
      {totalPages > 1 && (
        <div className="flex flex-col sm:flex-row justify-center items-center gap-3 mt-8">
          <button
            onClick={() => handlePageChange(page - 1)}
            disabled={page === 1}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg disabled:opacity-50 hover:bg-blue-700 transition-colors w-full sm:w-auto shadow-md"
          >
            السابق
          </button>
          <span className="text-gray-700 text-sm sm:text-base">
            صفحة {page} من {totalPages}
          </span>
          <button
            onClick={() => handlePageChange(page + 1)}
            disabled={page === totalPages}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg disabled:opacity-50 hover:bg-blue-700 transition-colors w-full sm:w-auto shadow-md"
          >
            التالي
          </button>
        </div>
      )}

      {/* Create Post Modal */}
      {isCreateModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50 px-4 py-6">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsCreateModalOpen(false)} />
          <div className="relative bg-white p-6 rounded-lg w-full max-w-2xl shadow-lg max-h-[90vh] overflow-y-auto">
            <h3 className="text-xl font-bold mb-4 text-gray-800 font-heading">إضافة منشور جديد</h3>
            <form onSubmit={createForm.handleSubmit(handleCreateSubmit)} className="space-y-6" dir="rtl">
              {uploadError && <p className="text-red-500 text-sm mb-4 bg-red-50 p-2 rounded">{uploadError}</p>}
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">العنوان</label>
                <input
                  {...createForm.register('title')}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm"
                />
                {createForm.formState.errors.title && <p className="text-red-500 text-xs mt-1">{createForm.formState.errors.title.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الوصف</label>
                <textarea
                  {...createForm.register('description')}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm min-h-[100px]"
                />
                {createForm.formState.errors.description && <p className="text-red-500 text-xs mt-1">{createForm.formState.errors.description.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الفئة</label>
                <select
                  {...createForm.register('category_id', { valueAsNumber: true })}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm"
                >
                  <option value="">اختر فئة</option>
                  {subCategories?.map((cat) => (
                    <option key={cat.id} value={cat.id}>
                      {cat.name}
                    </option>
                  ))}
                </select>
                {createForm.formState.errors.category_id && <p className="text-red-500 text-xs mt-1">{createForm.formState.errors.category_id.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الصورة المصغرة</label>
                <label className="w-full px-4 py-2 border border-gray-300 rounded-lg flex items-center gap-2 cursor-pointer text-sm bg-white shadow-sm hover:bg-gray-50 transition-colors">
                  <FaImage className="text-gray-600" /> اختر صورة
                  <input type="file" accept="image/*" className="hidden" onChange={(e) => handleThumbnailChange(e, createForm)} />
                </label>
                {isThumbnailLoading ? (
                  <div className="mt-2 w-24 sm:w-32 h-24 sm:h-32 flex items-center justify-center">
                    <div className="animate-spin rounded-full h-8 w-8 border-t-2 border-b-2 border-blue-500"></div>
                  </div>
                ) : thumbnailPreview ? (
                  <div className="mt-2">
                    <Image width={120} height={120} src={thumbnailPreview} alt="Preview" className="w-24 h-24 sm:w-32 sm:h-32 object-cover rounded-lg shadow-sm" />
                  </div>
                ) : null}
                {createForm.formState.errors.thumbnail && <p className="text-red-500 text-xs mt-1">{createForm.formState.errors.thumbnail.message}</p>}
              </div>
              <div>
                <label className="block mb-2 text-sm font-medium text-gray-700">المحتوى</label>
                <div className="flex flex-wrap gap-3 mb-4">
                  {[
                    { icon: FaParagraph, label: 'فقرة', type: 'paragraph' },
                    { icon: FaHeading, label: 'عنوان فرعي', type: 'subtitle' },
                    { icon: FaListUl, label: 'قائمة غير مرتبة', type: 'unordered-list' },
                    { icon: FaListOl, label: 'قائمة مرتبة', type: 'ordered-list' },
                    { icon: FaImage, label: 'صورة', type: 'image' },
                    { icon: FaYoutube, label: 'فيديو YouTube', type: 'youtube' },
                  ].map(({ icon: Icon, label, type }) => (
                    <button
                      key={type}
                      type="button"
                      onClick={() => addContentBlock(createForm, createAppend, type as any)}
                      className="relative p-2 bg-gray-200 rounded-lg hover:bg-gray-300 transition-colors group"
                    >
                      <Icon className="text-gray-700" size={20} />
                      <span className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 hidden group-hover:block text-xs bg-gray-800 text-white py-1 px-2 rounded shadow-lg">
                        {label}
                      </span>
                    </button>
                  ))}
                </div>
                <DndContext sensors={sensors} collisionDetection={closestCenter} onDragEnd={handleCreateDragEnd}>
                  <SortableContext items={createFields.map((field) => field.id)} strategy={verticalListSortingStrategy}>
                    {createFields.map((field, index) => (
                      <SortableItem key={field.id} id={field.id} field={field} index={index} form={createForm} remove={createRemove} />
                    ))}
                  </SortableContext>
                </DndContext>
                {createForm.formState.errors.content && !Array.isArray(createForm.formState.errors.content) && (
                  <p className="text-red-500 text-xs mt-1">{createForm.formState.errors.content.message}</p>
                )}
              </div>
              <div className="flex gap-2">
                <button
                  type="submit"
                  disabled={createMutation.isPending || isThumbnailLoading || contentImageLoadingIndices.length > 0}
                  className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors shadow-md"
                >
                  {createMutation.isPending ? 'جارٍ...' : 'إضافة'}
                </button>
                <button
                  type="button"
                  onClick={() => setIsCreateModalOpen(false)}
                  className="flex-1 px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400 transition-colors shadow-md"
                >
                  إلغاء
                </button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* Edit Post Modal */}
      {isEditModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50 px-4 py-6">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsEditModalOpen(false)} />
          <div className="relative bg-white p-6 rounded-lg w-full max-w-2xl shadow-lg max-h-[90vh] overflow-y-auto">
            <h3 className="text-xl font-bold mb-4 text-gray-800 font-heading">تعديل المنشور</h3>
            <form onSubmit={updateForm.handleSubmit(handleUpdateSubmit)} className="space-y-6" dir="rtl">
              {uploadError && <p className="text-red-500 text-sm mb-4 bg-red-50 p-2 rounded">{uploadError}</p>}
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">العنوان</label>
                <input
                  {...updateForm.register('title')}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm"
                />
                {updateForm.formState.errors.title && <p className="text-red-500 text-xs mt-1">{updateForm.formState.errors.title.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الوصف</label>
                <textarea
                  {...updateForm.register('description')}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm min-h-[100px]"
                />
                {updateForm.formState.errors.description && <p className="text-red-500 text-xs mt-1">{updateForm.formState.errors.description.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الفئة</label>
                <select
                  {...updateForm.register('category_id', { valueAsNumber: true })}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm"
                >
                  <option value="">اختر فئة</option>
                  {subCategories?.map((cat) => (
                    <option key={cat.id} value={cat.id}>
                      {cat.name}
                    </option>
                  ))}
                </select>
                {updateForm.formState.errors.category_id && <p className="text-red-500 text-xs mt-1">{updateForm.formState.errors.category_id.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الصورة المصغرة</label>
                <label className="w-full px-4 py-2 border border-gray-300 rounded-lg flex items-center gap-2 cursor-pointer text-sm bg-white shadow-sm hover:bg-gray-50 transition-colors">
                  <FaImage className="text-gray-600" /> اختر صورة
                  <input type="file" accept="image/*" className="hidden" onChange={(e) => handleThumbnailChange(e, updateForm)} />
                </label>
                {isThumbnailLoading ? (
                  <div className="mt-2 w-24 sm:w-32 h-24 sm:h-32 flex items-center justify-center">
                    <div className="animate-spin rounded-full h-8 w-8 border-t-2 border-b-2 border-blue-500"></div>
                  </div>
                ) : thumbnailPreview ? (
                  <div className="mt-2">
                    <Image width={120} height={120} src={thumbnailPreview} alt="Preview" className="w-24 h-24 sm:w-32 sm:h-32 object-cover rounded-lg shadow-sm" />
                  </div>
                ) : null}
                {updateForm.formState.errors.thumbnail && <p className="text-red-500 text-xs mt-1">{updateForm.formState.errors.thumbnail.message}</p>}
              </div>
              <div>
                <label className="block mb-2 text-sm font-medium text-gray-700">المحتوى</label>
                <div className="flex flex-wrap gap-3 mb-4">
                  {[
                    { icon: FaParagraph, label: 'فقرة', type: 'paragraph' },
                    { icon: FaHeading, label: 'عنوان فرعي', type: 'subtitle' },
                    { icon: FaListUl, label: 'قائمة غير مرتبة', type: 'unordered-list' },
                    { icon: FaListOl, label: 'قائمة مرتبة', type: 'ordered-list' },
                    { icon: FaImage, label: 'صورة', type: 'image' },
                    { icon: FaYoutube, label: 'فيديو YouTube', type: 'youtube' },
                  ].map(({ icon: Icon, label, type }) => (
                    <button
                      key={type}
                      type="button"
                      onClick={() => addContentBlock(updateForm, updateAppend, type as any)}
                      className="relative p-2 bg-gray-200 rounded-lg hover:bg-gray-300 transition-colors group"
                    >
                      <Icon className="text-gray-700" size={20} />
                      <span className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 hidden group-hover:block text-xs bg-gray-800 text-white py-1 px-2 rounded shadow-lg">
                        {label}
                      </span>
                    </button>
                  ))}
                </div>
                <DndContext sensors={sensors} collisionDetection={closestCenter} onDragEnd={handleUpdateDragEnd}>
                  <SortableContext items={updateFields.map((field) => field.id)} strategy={verticalListSortingStrategy}>
                    {updateFields.map((field, index) => (
                      <SortableItem key={field.id} id={field.id} field={field} index={index} form={updateForm} remove={updateRemove} />
                    ))}
                  </SortableContext>
                </DndContext>
                {updateForm.formState.errors.content && !Array.isArray(updateForm.formState.errors.content) && (
                  <p className="text-red-500 text-xs mt-1">{updateForm.formState.errors.content.message}</p>
                )}
              </div>
              <div className="flex gap-2">
                <button
                  type="submit"
                  disabled={updateMutation.isPending || isThumbnailLoading || contentImageLoadingIndices.length > 0}
                  className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors shadow-md"
                >
                  {updateMutation.isPending ? 'جارٍ...' : 'حفظ'}
                </button>
                <button
                  type="button"
                  onClick={() => setIsEditModalOpen(false)}
                  className="flex-1 px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400 transition-colors shadow-md"
                >
                  إلغاء
                </button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* Create Review Modal */}
      {isReviewModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50 px-4 py-6">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsReviewModalOpen(false)} />
          <div className="relative bg-white p-6 rounded-lg w-full max-w-md shadow-lg">
            <h3 className="text-xl font-bold mb-4 text-gray-800 font-heading">إضافة تقييم للمنشور</h3>
            <form onSubmit={reviewForm.handleSubmit(handleReviewSubmit)} className="space-y-4" dir="rtl">
              {uploadError && <p className="text-red-500 text-sm mb-4 bg-red-50 p-2 rounded">{uploadError}</p>}
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الاسم</label>
                <input
                  {...reviewForm.register('name')}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm"
                  placeholder="الاسم"
                />
                {reviewForm.formState.errors.name && <p className="text-red-500 text-xs mt-1">{reviewForm.formState.errors.name.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">النص</label>
                <textarea
                  {...reviewForm.register('text')}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm min-h-[100px]"
                  placeholder="النص"
                />
                {reviewForm.formState.errors.text && <p className="text-red-500 text-xs mt-1">{reviewForm.formState.errors.text.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">التقييم (1-5)</label>
                <input
                  type="number"
                  {...reviewForm.register('rate', { valueAsNumber: true })}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm"
                  placeholder="التقييم"
                  min={1}
                  max={5}
                />
                {reviewForm.formState.errors.rate && <p className="text-red-500 text-xs mt-1">{reviewForm.formState.errors.rate.message}</p>}
              </div>
              <div>
                <label className="flex items-center gap-2 text-sm font-medium text-gray-700">
                  <input type="checkbox" {...reviewForm.register('is_active')} className="rounded border-gray-300 text-blue-600 focus:ring-blue-500" />
                  نشط
                </label>
                {reviewForm.formState.errors.is_active && <p className="text-red-500 text-xs mt-1">{reviewForm.formState.errors.is_active.message}</p>}
              </div>
              <div className="flex gap-2">
                <button
                  type="submit"
                  disabled={createReviewMutation.isPending}
                  className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors shadow-md"
                >
                  {createReviewMutation.isPending ? 'جارٍ...' : 'إضافة'}
                </button>
                <button
                  type="button"
                  onClick={() => setIsReviewModalOpen(false)}
                  className="flex-1 px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400 transition-colors shadow-md"
                >
                  إلغاء
                </button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* Edit Review Modal */}
      {isEditReviewModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-55 px-4 py-6">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsEditReviewModalOpen(false)} />
          <div className="relative bg-white p-6 rounded-lg w-full max-w-md shadow-lg">
            <h3 className="text-xl font-bold mb-4 text-gray-800 font-heading">تعديل التقييم</h3>
            <form onSubmit={editReviewForm.handleSubmit(handleUpdateReviewSubmit)} className="space-y-4" dir="rtl">
              {uploadError && <p className="text-red-500 text-sm mb-4 bg-red-50 p-2 rounded">{uploadError}</p>}
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الاسم</label>
                <input
                  {...editReviewForm.register('name')}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm"
                  placeholder="الاسم"
                />
                {editReviewForm.formState.errors.name && <p className="text-red-500 text-xs mt-1">{editReviewForm.formState.errors.name.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">النص</label>
                <textarea
                  {...editReviewForm.register('text')}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm min-h-[100px]"
                  placeholder="النص"
                />
                {editReviewForm.formState.errors.text && <p className="text-red-500 text-xs mt-1">{editReviewForm.formState.errors.text.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">التقييم (1-5)</label>
                <input
                  type="number"
                  {...editReviewForm.register('rate', { valueAsNumber: true })}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm"
                  placeholder="التقييم"
                  min={1}
                  max={5}
                />
                {editReviewForm.formState.errors.rate && <p className="text-red-500 text-xs mt-1">{editReviewForm.formState.errors.rate.message}</p>}
              </div>
              <div>
                <label className="flex items-center gap-2 text-sm font-medium text-gray-700">
                  <input type="checkbox" {...editReviewForm.register('is_active')} className="rounded border-gray-300 text-blue-600 focus:ring-blue-500" />
                  نشط
                </label>
                {editReviewForm.formState.errors.is_active && <p className="text-red-500 text-xs mt-1">{editReviewForm.formState.errors.is_active.message}</p>}
              </div>
              <div className="flex gap-2">
                <button
                  type="submit"
                  disabled={updateReviewMutation.isPending}
                  className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors shadow-md"
                >
                  {updateReviewMutation.isPending ? 'جارٍ...' : 'حفظ'}
                </button>
                <button
                  type="button"
                  onClick={() => setIsEditReviewModalOpen(false)}
                  className="flex-1 px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400 transition-colors shadow-md"
                >
                  إلغاء
                </button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* View Reviews Modal */}
      {isViewReviewsModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50 px-4 py-6">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsViewReviewsModalOpen(false)} />
          <div className="relative bg-white p-6 rounded-lg w-full max-w-lg shadow-lg max-h-[60vh] overflow-y-auto">
            <h3 className="text-xl font-bold mb-4 text-gray-800 font-heading">تقييمات المنشور</h3>
            <div dir="rtl">
              <div className="mb-4">
                <label className="flex items-center gap-2 text-sm font-medium text-gray-700">
                  تصفية حسب الحالة:
                  <select
                    value={reviewFilter}
                    onChange={(e) => setReviewFilter(e.target.value as 'all' | 'active' | 'in_active')}
                    className="px-4 py-2 border border-gray-300 rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 shadow-sm"
                  >
                    <option value="all">الكل</option>
                    <option value="active">نشط</option>
                    <option value="in_active">غير نشط</option>
                  </select>
                </label>
              </div>
              {reviewsLoading ? (
                <div className="text-center py-4 text-gray-600">جارٍ التحميل...</div>
              ) : filteredReviews.length > 0 ? (
                <>
                  {uploadError && <p className="text-red-500 text-sm mb-4 bg-red-50 p-2 rounded">{uploadError}</p>}
                  <div className="space-y-4">
                    {filteredReviews.map((review: any) => (
                      <div key={review.id} className="flex items-start gap-4 p-4 border border-neutral-200 rounded-lg">
                        <input type="checkbox" checked={selectedReviewIds.includes(review.id)} onChange={() => toggleReviewSelection(review.id)} className="mt-1" />
                        <div className="flex-1">
                          <div className="flex items-center gap-2">
                            <p className="font-semibold text-gray-800">{review.name}</p>
                            <div className="flex">
                              {[...Array(5)].map((_, i) => (
                                <FaStar key={i} className={i < review.rate ? 'text-yellow-400' : 'text-gray-300'} />
                              ))}
                            </div>
                            <span className={`text-sm px-2 py-1 rounded-full ${review.is_active ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'}`}>
                              {review.is_active ? 'نشط' : 'غير نشط'}
                            </span>
                          </div>
                          <p className="text-gray-600">{review.text}</p>
                          <p className="text-xs text-gray-500">{new Date(review.createdAt).toLocaleDateString('ar')}</p>
                          <button
                            onClick={() => handleEditReviewClick(review)}
                            className="mt-2 bg-blue-600 text-white text-sm font-medium px-3 py-1.5 rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 transition duration-200 ease-in-out"
                          >
                            تعديل
                          </button>
                        </div>
                      </div>
                    ))}
                  </div>
                  {selectedReviewIds.length > 0 && (
                    <button
                      onClick={handleDeleteReviews}
                      disabled={deleteReviewMutation.isPending}
                      className="mt-4 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 disabled:opacity-50 transition-colors flex items-center justify-center gap-2 w-full shadow-md"
                    >
                      <FaTrash /> حذف المحدد ({selectedReviewIds.length})
                    </button>
                  )}
                </>
              ) : (
                <p className="text-center text-gray-600">لا توجد تقييمات لهذا المنشور</p>
              )}
            </div>
            <div className="mt-4">
              <button
                onClick={() => setIsViewReviewsModalOpen(false)}
                className="w-full px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400 transition-colors shadow-md"
              >
                إغلاق
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default Posts;
