import Image from 'next/image';
import React from 'react';

interface ContentBlock {
  type: 'paragraph' | 'subtitle' | 'unordered-list' | 'ordered-list' | 'image' | 'youtube';
  value: string | string[];
}

interface ContentRendererProps {
  content: ContentBlock[];
}

const getYouTubeEmbedUrl = (url: string): string => {
  const match = url.match(/(?:https?:\/\/)?(?:www\.)?youtube\.com\/watch\?v=([a-zA-Z0-9_-]+)/) || url.match(/(?:https?:\/\/)?(?:www\.)?youtu\.be\/([a-zA-Z0-9_-]+)/);
  if (match && match[1]) {
    return `https://www.youtube.com/embed/${match[1]}`;
  }
  return url; // fallback to original if it doesn't match
};

const ContentRenderer: React.FC<ContentRendererProps> = ({ content }) => {
  return (
    <div className="space-y-6">
      {content.map((block, index) => {
        switch (block.type) {
          case 'paragraph':
            return (
              <p key={index} className="text-gray-600 text-base font-body leading-relaxed">
                {block.value as string}
              </p>
            );
          case 'subtitle':
            return (
              <h2 key={index} className="text-xl md:text-2xl font-semibold text-black font-heading mt-8 mb-4">
                {block.value as string}
              </h2>
            );
          case 'unordered-list':
            return (
              <ul key={index} className="list-disc list-inside text-gray-600 text-base font-body leading-relaxed space-y-2">
                {(block.value as string[]).map((item, i) => (
                  <li key={i}>{item}</li>
                ))}
              </ul>
            );
          case 'ordered-list':
            return (
              <ol key={index} className="list-decimal list-inside text-gray-600 text-base font-body leading-relaxed space-y-2">
                {(block.value as string[]).map((item, i) => (
                  <li key={i}>{item}</li>
                ))}
              </ol>
            );
          case 'image':
            return (
              <div key={index} className="relative w-full max-w-md h-48 md:h-64 mx-auto">
                <Image src={block.value as string} alt={`Image ${index + 1}`} fill className="object-cover rounded-md" />
              </div>
            );
          case 'youtube':
            return (
              <div key={index} className="relative w-full max-w-md aspect-video mx-auto">
                <iframe
                  src={getYouTubeEmbedUrl(block.value as string)}
                  title={`YouTube Video ${index + 1}`}
                  className="w-full h-full rounded-md"
                  allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                  allowFullScreen
                />
              </div>
            );
          default:
            return null;
        }
      })}
    </div>
  );
};

export default ContentRenderer;
