/* eslint-disable @typescript-eslint/no-explicit-any */

'use client';

import { zodResolver } from '@hookform/resolvers/zod';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { FaBook, FaEdit, FaImage, FaList, FaPlus, FaTrash, FaVideo } from 'react-icons/fa';
import { FaNoteSticky } from 'react-icons/fa6';
import { createMediaFormSchema, CreateMediaFormValues, Media_Types, updateMediaFormSchema, UpdateMediaFormValues } from './media.schema';
import { createMedia, deleteMedia, getMediaDetails, listMedia, updateMedia, uploadFile, removeFile } from './media.service';
import Image from 'next/image';
import Link from 'next/link';

const Media = () => {
  const queryClient = useQueryClient();
  const [filterType, setFilterType] = useState<string | undefined>(undefined); // ALL by default
  const [page, setPage] = useState(1);
  const [selectedIds, setSelectedIds] = useState<number[]>([]);
  const [isCreateModalOpen, setIsCreateModalOpen] = useState(false);
  const [isEditModalOpen, setIsEditModalOpen] = useState(false);
  const [editingMediaId, setEditingMediaId] = useState<number | null>(null);
  const [thumbnailPreview, setThumbnailPreview] = useState<string | null>(null);
  const [uploadError, setUploadError] = useState<string | null>(null);
  const [isThumbnailLoading, setIsThumbnailLoading] = useState(false); // Loading state for thumbnail
  const pageSize = 10;

  // Fetch media with filter and pagination
  const { data, isLoading, error } = useQuery({
    queryKey: ['media', filterType, page],
    queryFn: () => listMedia({ page, pageSize }, filterType),
  });

  const mediaItems = data?.data || [];
  const totalItems = data?.pagination.totalItemsCount || 0;
  const totalPages = Math.ceil(totalItems / pageSize);

  // Form setup for create
  const createForm = useForm<CreateMediaFormValues>({
    resolver: zodResolver(createMediaFormSchema),
    defaultValues: {
      title: '',
      description: '',
      thumbnail: [],
      url: '',
      media_type: Media_Types.VIDEO,
      is_active: true,
    },
  });

  // Form setup for update
  const updateForm = useForm<UpdateMediaFormValues>({
    resolver: zodResolver(updateMediaFormSchema),
    defaultValues: {
      title: '',
      description: '',
      thumbnail: [],
      url: '',
      media_type: Media_Types.VIDEO,
    },
  });

  // Create mutation
  const createMutation = useMutation({
    mutationFn: createMedia,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['media'] });
      setIsCreateModalOpen(false);
      setThumbnailPreview(null);
      setUploadError(null);
      setIsThumbnailLoading(false);
      createForm.reset();
    },
    onError: (err: any) => {
      console.error('Create media error:', err);
      setUploadError('فشل في إنشاء الوسائط: ' + (err.message || 'خطأ غير معروف'));
    },
  });

  // Update mutation
  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: UpdateMediaFormValues }) => {
      const payload = { ...data, is_active: true };
      return updateMedia(id, payload);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['media'] });
      setIsEditModalOpen(false);
      setEditingMediaId(null);
      setThumbnailPreview(null);
      setUploadError(null);
      setIsThumbnailLoading(false);
      updateForm.reset();
    },
    onError: (err: any) => {
      console.error('Update media error:', err);
      setUploadError('فشل في تحديث الوسائط: ' + (err.message || 'خطأ غير معروف'));
    },
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: deleteMedia,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['media'] });
      setSelectedIds([]);
    },
    onError: (err: any) => {
      console.error('Delete media error:', err);
      setUploadError('فشل في حذف الوسائط: ' + (err.message || 'خطأ غير معروف'));
    },
  });

  const handleFilterChange = (type: string | undefined) => {
    setFilterType(type);
    setPage(1);
    setSelectedIds([]);
  };

  const handlePageChange = (newPage: number) => {
    if (newPage >= 1 && newPage <= totalPages) {
      setPage(newPage);
      setSelectedIds([]);
    }
  };

  const handleSelectMedia = (id: number) => {
    setSelectedIds((prev) => (prev.includes(id) ? prev.filter((item) => item !== id) : [...prev, id]));
  };

  const handleCreateSubmit = (data: CreateMediaFormValues) => {
    createMutation.mutate(data);
  };

  const handleEditClick = async (id: number) => {
    try {
      const media = await getMediaDetails(id);
      setEditingMediaId(id);
      updateForm.reset({
        title: media.title,
        description: media.description || '',
        thumbnail: media.thumbnail || [],
        url: media.url || '',
        media_type: media.media_type || Media_Types.VIDEO,
      });
      setThumbnailPreview(media.thumbnail?.[0]?.url || null);
      setIsEditModalOpen(true);
      setUploadError(null);
      setIsThumbnailLoading(false);
    } catch (err) {
      console.error('Error fetching media details:', err);
      setUploadError('فشل في جلب تفاصيل الوسائط');
    }
  };

  const handleUpdateSubmit = (data: UpdateMediaFormValues) => {
    if (editingMediaId) {
      updateMutation.mutate({ id: editingMediaId, data });
    }
  };

  const handleDeleteSelected = () => {
    if (selectedIds.length > 0) {
      deleteMutation.mutate(selectedIds);
    }
  };

  const handleThumbnailChange = async (e: React.ChangeEvent<HTMLInputElement>, form: any) => {
    const file = e.target.files?.[0];
    if (file) {
      try {
        setIsThumbnailLoading(true); // Show spinner
        setUploadError(null);

        // Attempt to remove the old thumbnail (if it exists), but don't block upload
        const existingThumbnail = form.getValues('thumbnail');
        if (existingThumbnail && existingThumbnail.length > 0) {
          const oldFileUrl = existingThumbnail[0].url;
          removeFile(oldFileUrl);
          // .then(() => console.log(`Successfully removed old thumbnail: ${oldFileUrl}`))
          // .catch((err) => console.error(`Failed to remove old thumbnail: ${oldFileUrl}`, err));
          // Note: We don’t await here, so removal runs in the background
        }

        // Upload the new thumbnail regardless of removal success/failure
        const uploadResponse = await uploadFile(file);
        const thumbnailData = uploadResponse.data.files;
        if (thumbnailData.length === 0) {
          throw new Error('No thumbnail data returned');
        }
        form.setValue('thumbnail', thumbnailData);
        setThumbnailPreview(thumbnailData[0].url);
      } catch (err: any) {
        console.error('Thumbnail upload error:', err);
        setUploadError('فشل في رفع الصورة: ' + (err.message || 'خطأ غير معروف'));
        form.setValue('thumbnail', []);
      } finally {
        setIsThumbnailLoading(false); // Hide spinner
      }
    } else {
      setUploadError('يرجى اختيار صورة مصغرة');
      form.setValue('thumbnail', []);
      setIsThumbnailLoading(false);
    }
  };

  const openCreateModal = () => {
    createForm.reset();
    setThumbnailPreview(null);
    setUploadError(null);
    setIsThumbnailLoading(false);
    setIsCreateModalOpen(true);
  };

  return (
    <div className="px-4 sm:px-6 py-4 min-h-screen" dir="rtl">
      <h2 className="text-xl sm:text-2xl font-bold mb-4 sm:mb-6 text-doctor-dark font-heading text-start">إدارة الوسائط</h2>

      {/* Filter and Actions */}
      <div className="flex flex-col sm:flex-row gap-4 mb-4 sm:mb-6 items-center justify-between">
        <div className="flex flex-wrap gap-2 sm:gap-4">
          <button
            onClick={() => handleFilterChange(undefined)}
            className={`px-3 py-2 rounded-lg flex items-center gap-2 text-sm sm:text-base ${
              !filterType ? 'bg-doctor-dark text-white' : 'bg-gray-200 text-gray-700'
            } hover:bg-doctor-primary transition-colors`}
          >
            <FaList /> الكل
          </button>
          <button
            onClick={() => handleFilterChange(Media_Types.VIDEO)}
            className={`px-3 py-2 rounded-lg flex items-center gap-2 text-sm sm:text-base ${
              filterType === Media_Types.VIDEO ? 'bg-doctor-dark text-white' : 'bg-gray-200 text-gray-700'
            } hover:bg-doctor-primary transition-colors`}
          >
            <FaVideo /> فيديوهات
          </button>
          <button
            onClick={() => handleFilterChange(Media_Types.LECTURE)}
            className={`px-3 py-2 rounded-lg flex items-center gap-2 text-sm sm:text-base ${
              filterType === Media_Types.LECTURE ? 'bg-doctor-dark text-white' : 'bg-gray-200 text-gray-700'
            } hover:bg-doctor-primary transition-colors`}
          >
            <FaBook /> محاضرات
          </button>
          <button
            onClick={() => handleFilterChange(Media_Types.REPORT)}
            className={`px-3 py-2 rounded-lg flex items-center gap-2 text-sm sm:text-base ${
              filterType === Media_Types.REPORT ? 'bg-doctor-dark text-white' : 'bg-gray-200 text-gray-700'
            } hover:bg-doctor-primary transition-colors`}
          >
            <FaNoteSticky /> تقارير
          </button>
        </div>
        <div className="flex flex-col sm:flex-row gap-2 sm:gap-4 w-full sm:w-auto">
          <button
            onClick={openCreateModal}
            className="px-3 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary flex items-center justify-center gap-2 w-full sm:w-auto text-sm sm:text-base"
          >
            <FaPlus /> إضافة وسائط
          </button>
          {selectedIds.length > 0 && (
            <button
              onClick={handleDeleteSelected}
              className="px-3 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 flex items-center justify-center gap-2 w-full sm:w-auto text-sm sm:text-base"
              disabled={deleteMutation.isPending}
            >
              <FaTrash /> حذف ({selectedIds.length})
            </button>
          )}
        </div>
      </div>

      {/* Media List */}
      {isLoading ? (
        <div className="text-center py-8 text-gray-600 text-sm sm:text-lg">جارٍ التحميل...</div>
      ) : error ? (
        <div className="text-center py-8 text-red-500 text-sm sm:text-lg">حدث خطأ أثناء جلب البيانات: {JSON.stringify(error)}</div>
      ) : mediaItems.length > 0 ? (
        <div className="space-y-4 sm:space-y-6">
          {mediaItems.map((media) => (
            <div
              key={media.id}
              className={`flex flex-col sm:flex-row items-center gap-4 sm:gap-6 p-4 sm:p-6 bg-white border border-neutral-200 rounded-lg hover:shadow-sm transition-all duration-300 ${
                selectedIds.includes(media.id) ? 'border-doctor-dark border-2' : ''
              }`}
            >
              {/* Thumbnail */}
              <Link href={media.url} target="_blank">
                {media.thumbnail && media.thumbnail.length > 0 ? (
                  <div className="relative w-full sm:w-32 md:w-40 h-32 md:h-40 rounded-lg overflow-hidden">
                    <Image width={1000} height={1000} src={media.thumbnail[0].url} alt={media.title} className="w-full h-full object-cover" />
                    <div className="absolute inset-0 flex items-center justify-center">
                      <div className="bg-white/80 rounded-full p-2 sm:p-3">
                        <svg className="w-6 h-6 sm:w-8 sm:h-8 text-doctor-primary" fill="currentColor" viewBox="0 0 24 24">
                          <path d="M8 5v14l11-7z" />
                        </svg>
                      </div>
                    </div>
                  </div>
                ) : (
                  <div className="w-full sm:w-32 md:w-40 h-32 md:h-40 bg-gray-200 rounded-lg flex items-center justify-center">
                    <span className="text-gray-500 text-sm sm:text-base">بدون صورة</span>
                  </div>
                )}
              </Link>

              {/* Text Content */}
              <div className="flex-1 w-full">
                <div className="flex items-center justify-between gap-2">
                  <h3 className="text-lg sm:text-xl font-semibold text-doctor-dark font-heading mb-2 truncate w-full sm:w-auto">{media.title}</h3>
                  <div className="flex gap-3 items-center">
                    <button onClick={() => handleEditClick(media.id)} className="text-doctor-dark hover:text-doctor-primary">
                      <FaEdit size={18} />
                    </button>
                    <input
                      type="checkbox"
                      className="h-4 w-4 sm:h-5 sm:w-5 text-blue-600 rounded focus:ring-blue-500"
                      checked={selectedIds.includes(media.id)}
                      onChange={() => handleSelectMedia(media.id)}
                    />
                  </div>
                </div>
                <p className="text-gray-600 text-sm sm:text-base leading-relaxed mb-2 sm:mb-4 line-clamp-2">{media.description || 'بدون وصف'}</p>
                <span
                  className={`px-2 sm:px-3 py-1 text-xs sm:text-sm font-semibold rounded-full ${
                    media.media_type === 'video'
                      ? 'bg-red-100 text-red-600'
                      : media.media_type === 'lecture'
                      ? 'bg-sky-100 text-sky-600'
                      : media.media_type === 'report'
                      ? 'bg-amber-100 text-amber-600'
                      : 'bg-gray-100 text-gray-600'
                  }`}
                >
                  {media.media_type === 'video' ? 'فيديو' : media.media_type === 'lecture' ? 'محاضرة' : media.media_type === 'report' ? 'تقرير' : 'غير محدد'}
                </span>
              </div>
            </div>
          ))}
        </div>
      ) : (
        <div className="text-center py-8 text-gray-600 text-sm sm:text-lg">لا توجد وسائط</div>
      )}

      {/* Pagination */}
      {totalPages > 1 && (
        <div className="flex flex-col sm:flex-row justify-center items-center gap-2 sm:gap-4 mt-4 sm:mt-6">
          <button
            onClick={() => handlePageChange(page - 1)}
            disabled={page === 1}
            className="px-3 py-2 bg-doctor-dark text-white rounded-lg disabled:opacity-50 hover:bg-doctor-primary transition-colors w-full sm:w-auto text-sm sm:text-base"
          >
            السابق
          </button>
          <span className="text-gray-700 text-sm sm:text-base">
            صفحة {page} من {totalPages}
          </span>
          <button
            onClick={() => handlePageChange(page + 1)}
            disabled={page === totalPages}
            className="px-3 py-2 bg-doctor-dark text-white rounded-lg disabled:opacity-50 hover:bg-doctor-primary transition-colors w-full sm:w-auto text-sm sm:text-base"
          >
            التالي
          </button>
        </div>
      )}

      {/* Create Modal */}
      {isCreateModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50 px-4 sm:px-0">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsCreateModalOpen(false)} />
          <div className="relative bg-white p-4 sm:p-6 md:p-8 rounded-xl w-full max-w-md shadow-2xl">
            <h3 className="text-lg sm:text-xl font-bold mb-4 text-doctor-dark">إضافة وسائط جديدة</h3>
            {uploadError && <p className="text-red-500 text-xs sm:text-sm mb-4">{uploadError}</p>}
            <form onSubmit={createForm.handleSubmit(handleCreateSubmit)} className="space-y-4" dir="rtl">
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">العنوان</label>
                <input
                  {...createForm.register('title')}
                  className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark text-sm sm:text-base"
                />
                {createForm.formState.errors.title && <p className="text-red-500 text-xs sm:text-sm mt-1">{createForm.formState.errors.title.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الوصف</label>
                <textarea
                  {...createForm.register('description')}
                  className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark text-sm sm:text-base"
                />
                {createForm.formState.errors.description && <p className="text-red-500 text-xs sm:text-sm mt-1">{createForm.formState.errors.description.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">النوع</label>
                <select
                  {...createForm.register('media_type')}
                  className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark text-sm sm:text-base"
                >
                  <option value={Media_Types.VIDEO}>فيديو</option>
                  <option value={Media_Types.LECTURE}>محاضرة</option>
                  <option value={Media_Types.REPORT}>تقرير</option>
                </select>
                {createForm.formState.errors.media_type && <p className="text-red-500 text-xs sm:text-sm mt-1">{createForm.formState.errors.media_type.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">رابط URL</label>
                <input
                  {...createForm.register('url')}
                  className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark text-sm sm:text-base"
                />
                {createForm.formState.errors.url && <p className="text-red-500 text-xs sm:text-sm mt-1">{createForm.formState.errors.url.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الصورة المصغرة</label>
                <label className="w-full px-3 py-2 border rounded-lg flex items-center gap-2 cursor-pointer text-sm sm:text-base">
                  <FaImage /> اختر صورة
                  <input type="file" accept="image/*" className="hidden" onChange={(e) => handleThumbnailChange(e, createForm)} />
                </label>
                {isThumbnailLoading ? (
                  <div className="mt-2 w-24 sm:w-32 h-24 sm:h-32 flex items-center justify-center">
                    <div className="animate-spin rounded-full h-8 w-8 border-t-2 border-b-2 border-doctor-primary"></div>
                  </div>
                ) : thumbnailPreview ? (
                  <Image width={1000} height={1000} src={thumbnailPreview} alt="Preview" className="w-24 sm:w-32 h-24 sm:h-32 object-cover rounded-lg mt-2" />
                ) : null}
                {createForm.formState.errors.thumbnail && <p className="text-red-500 text-xs sm:text-sm mt-1">{createForm.formState.errors.thumbnail.message}</p>}
              </div>
              <div className="flex flex-col sm:flex-row gap-2 sm:gap-3">
                <button
                  type="submit"
                  disabled={createMutation.isPending || isThumbnailLoading}
                  className="flex-1 px-3 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary disabled:opacity-50 text-sm sm:text-base"
                >
                  {createMutation.isPending ? 'جارٍ...' : 'إضافة'}
                </button>
                <button
                  type="button"
                  onClick={() => setIsCreateModalOpen(false)}
                  className="flex-1 px-3 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400 text-sm sm:text-base"
                >
                  إلغاء
                </button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* Edit Modal */}
      {isEditModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50 px-4 sm:px-0">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsEditModalOpen(false)} />
          <div className="relative bg-white p-4 sm:p-6 md:p-8 rounded-xl w-full max-w-md shadow-2xl">
            <h3 className="text-lg sm:text-xl font-bold mb-4 text-doctor-dark">تعديل الوسائط</h3>
            {uploadError && <p className="text-red-500 text-xs sm:text-sm mb-4">{uploadError}</p>}
            <form onSubmit={updateForm.handleSubmit(handleUpdateSubmit)} className="space-y-4" dir="rtl">
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">العنوان</label>
                <input
                  {...updateForm.register('title')}
                  className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark text-sm sm:text-base"
                />
                {updateForm.formState.errors.title && <p className="text-red-500 text-xs sm:text-sm mt-1">{updateForm.formState.errors.title.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الوصف</label>
                <textarea
                  {...updateForm.register('description')}
                  className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark text-sm sm:text-base"
                />
                {updateForm.formState.errors.description && <p className="text-red-500 text-xs sm:text-sm mt-1">{updateForm.formState.errors.description.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">النوع</label>
                <select
                  {...updateForm.register('media_type')}
                  className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark text-sm sm:text-base"
                >
                  <option value={Media_Types.VIDEO}>فيديو</option>
                  <option value={Media_Types.LECTURE}>محاضرة</option>
                  <option value={Media_Types.REPORT}>تقرير</option>
                </select>
                {updateForm.formState.errors.media_type && <p className="text-red-500 text-xs sm:text-sm mt-1">{updateForm.formState.errors.media_type.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">رابط URL</label>
                <input
                  {...updateForm.register('url')}
                  className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark text-sm sm:text-base"
                />
                {updateForm.formState.errors.url && <p className="text-red-500 text-xs sm:text-sm mt-1">{updateForm.formState.errors.url.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الصورة المصغرة</label>
                <label className="w-full px-3 py-2 border rounded-lg flex items-center gap-2 cursor-pointer text-sm sm:text-base">
                  <FaImage /> اختر صورة
                  <input type="file" accept="image/*" className="hidden" onChange={(e) => handleThumbnailChange(e, updateForm)} />
                </label>
                {isThumbnailLoading ? (
                  <div className="mt-2 w-24 sm:w-32 h-24 sm:h-32 flex items-center justify-center">
                    <div className="animate-spin rounded-full h-8 w-8 border-t-2 border-b-2 border-doctor-primary"></div>
                  </div>
                ) : thumbnailPreview ? (
                  <Image width={1000} height={1000} src={thumbnailPreview} alt="Preview" className="w-24 sm:w-32 h-24 sm:h-32 object-cover rounded-lg mt-2" />
                ) : null}
                {updateForm.formState.errors.thumbnail && <p className="text-red-500 text-xs sm:text-sm mt-1">{updateForm.formState.errors.thumbnail.message}</p>}
              </div>
              <div className="flex flex-col sm:flex-row gap-2 sm:gap-3">
                <button
                  type="submit"
                  disabled={updateMutation.isPending || isThumbnailLoading}
                  className="flex-1 px-3 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary disabled:opacity-50 text-sm sm:text-base"
                >
                  {updateMutation.isPending ? 'جارٍ...' : 'حفظ'}
                </button>
                <button
                  type="button"
                  onClick={() => setIsEditModalOpen(false)}
                  className="flex-1 px-3 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400 text-sm sm:text-base"
                >
                  إلغاء
                </button>
              </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
};

export default Media;
