import Cookies from 'js-cookie';
import { CreateMediaFormValues, UpdateMediaFormValues, Media, PaginatedResponse, ListMediaFilter, FileMetadata } from './media.schema';

const API_URL = process.env.NEXT_PUBLIC_SERVER_URL || 'http://localhost:7000';

// Helper function to get the auth token from cookies
const getAuthToken = () => {
  if (typeof window !== 'undefined') {
    return Cookies.get('dr_auth_token');
  }
  return null;
};

// List media with pagination and optional type filter
export const listMedia = async (filter: ListMediaFilter, type?: string): Promise<PaginatedResponse<Media>> => {
  const queryParams = new URLSearchParams({
    page: filter.page.toString(),
    pageSize: filter.pageSize.toString(),
    ...(filter.q && { q: filter.q }),
    ...(filter.order && { order: JSON.stringify(filter.order) }), // Stringify order array
    ...(type && { type }),
  });

  const response = await fetch(`${API_URL}/media/list?${queryParams.toString()}`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as PaginatedResponse<Media>;
};

// Create a new media
export const createMedia = async (data: CreateMediaFormValues): Promise<Media> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/media/create`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Media;
};

// Get media details by ID
export const getMediaDetails = async (id: number): Promise<Media> => {
  const response = await fetch(`${API_URL}/media/${id}`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Media;
};

// Update media
export const updateMedia = async (id: number, data: UpdateMediaFormValues): Promise<Media> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/media/${id}`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Media;
};

// Delete multiple media
export const deleteMedia = async (ids: number[]): Promise<void> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/media/delete-many`, {
    method: 'DELETE',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify({ ids }),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result;
};

// Upload file (thumbnail)
export const uploadFile = async (file: File): Promise<{ message: string; data: { files: FileMetadata[] } }> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const formData = new FormData();
  formData.append('files', file);

  const response = await fetch(`${API_URL}/multerUploader/upload`, {
    method: 'POST',
    headers: {
      Authorization: `Bearer ${token}`,
    },
    body: formData,
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as { message: string; data: { files: FileMetadata[] } };
};

// Remove file (delete old image from server)
export const removeFile = async (fileUrl: string): Promise<void> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  await fetch(`${API_URL}/multerUploader/remove`, {
    method: 'DELETE',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify({
      files: [{ url: fileUrl }],
    }),
  });
};
