import { z } from 'zod';

// Media Types (aligned with backend Media_Types)
export enum Media_Types {
  VIDEO = 'video',
  LECTURE = 'lecture',
  REPORT = 'report', // Added to match backend
}

// FileMetadata for thumbnail
export interface FileMetadata {
  url: string;
  filename: string;
}

// Media interface based on migration (unchanged)
export interface Media {
  id: number;
  title: string;
  description: string | null;
  thumbnail: FileMetadata[] | null;
  url: string;
  media_type: Media_Types | null;
  index: number | null;
  is_active: boolean;
  createdAt: string;
  updatedAt: string;
}

// Schema for creating media (is_active required but always true)
export const createMediaFormSchema = z.object({
  title: z.string().min(1, 'العنوان مطلوب').max(256, 'العنوان يجب ألا يتجاوز 256 حرفًا'),
  description: z.string().min(1, 'الوصف مطلوب').max(1000, 'الوصف يجب ألا يتجاوز 1000 حرف'),
  thumbnail: z.array(z.object({ url: z.string(), filename: z.string() })).min(1, 'الصورة المصغرة مطلوبة'),
  url: z.string().min(1, 'رابط URL مطلوب').url('يجب أن يكون الرابط صالحًا'),
  media_type: z.nativeEnum(Media_Types, { message: 'نوع الوسائط مطلوب' }),
  is_active: z.literal(true), // Always true, no other value allowed
});

// Schema for updating media (is_active removed)
export const updateMediaFormSchema = z.object({
  title: z.string().min(1, 'العنوان مطلوب').max(256, 'العنوان يجب ألا يتجاوز 256 حرفًا'),
  description: z.string().min(1, 'الوصف مطلوب').max(1000, 'الوصف يجب ألا يتجاوز 1000 حرف'),
  thumbnail: z.array(z.object({ url: z.string(), filename: z.string() })).min(1, 'الصورة المصغرة مطلوبة'),
  url: z.string().min(1, 'رابط URL مطلوب').url('يجب أن يكون الرابط صالحًا'),
  media_type: z.nativeEnum(Media_Types, { message: 'نوع الوسائط مطلوب' }),
});

// Schema for listing media (pagination and filter, unchanged)
export const listMediaFilterSchema = z.object({
  page: z.number().min(1).optional().default(1),
  pageSize: z.number().min(1).optional().default(10),
  q: z.string().max(200).optional(),
  order: z
    .array(
      z.object({
        field: z.enum(['id', 'title', 'description', 'index']),
        orderType: z.enum(['asc', 'desc']),
      }),
    )
    .optional(),
});

// Types
export type CreateMediaFormValues = z.infer<typeof createMediaFormSchema>;
export type UpdateMediaFormValues = z.infer<typeof updateMediaFormSchema>;
export type ListMediaFilter = z.infer<typeof listMediaFilterSchema>;

// API response types
export interface PaginatedResponse<T> {
  data: T[];
  pagination: {
    totalItemsCount: number;
    pagesCount: number;
    page: string;
  };
}
