'use client';

import { handleBackendError } from '@/shared/api/errorHandler';
import { zodResolver } from '@hookform/resolvers/zod';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { useEffect, useState } from 'react';
import { useForm } from 'react-hook-form';
import { FaChevronDown, FaChevronUp, FaEdit, FaPlus, FaSpinner, FaTrash } from 'react-icons/fa';
import { CreateFAQFormValues, FAQ, UpdateFAQFormValues, createFAQFormSchema, updateFAQFormSchema } from './faq.schema';
import { createFAQ, deleteFAQs, getFAQDetails, listFAQs, updateFAQ } from './faq.service';

const FAQs = () => {
  const queryClient = useQueryClient();
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingFAQId, setEditingFAQId] = useState<number | null>(null);
  const [selectedFAQs, setSelectedFAQs] = useState<number[]>([]);
  const [openIndices, setOpenIndices] = useState<Set<number>>(new Set());

  // Form setup (use different schemas for create and update)
  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
    setValue,
  } = useForm<CreateFAQFormValues | UpdateFAQFormValues>({
    resolver: zodResolver(editingFAQId ? updateFAQFormSchema : createFAQFormSchema),
    defaultValues: {
      question: '',
      answer: '',
      index: 0,
      is_active: true,
    },
  });

  // Fetch FAQs
  const { data, isLoading, error } = useQuery({
    queryKey: ['faqs'],
    queryFn: () => listFAQs(),
  });

  // Initialize openIndices with all indices when data loads
  useEffect(() => {
    if (data?.length) {
      const indices = new Set<number>(data.map((_, index) => index));
      setOpenIndices(indices);
    }
  }, [data]);

  // Create FAQ mutation
  const createMutation = useMutation({
    mutationFn: createFAQ,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['faqs'] });
      setIsModalOpen(false);
      setEditingFAQId(null); // Ensure editingFAQId is reset
      reset();
    },
    onError: (err) => {
      console.error('Create error:', err);
    },
  });

  // Update FAQ mutation
  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: UpdateFAQFormValues }) => updateFAQ(id, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['faqs'] });
      setIsModalOpen(false);
      setEditingFAQId(null); // Ensure editingFAQId is reset
      reset();
    },
    onError: (err) => {
      console.error('Update error:', err);
    },
  });

  // Delete FAQs mutation
  const deleteMutation = useMutation({
    mutationFn: deleteFAQs,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['faqs'] });
      setSelectedFAQs([]);
    },
    onError: (err) => {
      console.error('Delete error:', err);
    },
  });

  // Handle form submission
  const onSubmit = (data: CreateFAQFormValues | UpdateFAQFormValues) => {
    if (editingFAQId) {
      updateMutation.mutate({ id: editingFAQId, data: data as UpdateFAQFormValues });
    } else {
      createMutation.mutate(data as CreateFAQFormValues);
    }
  };

  // Handle edit button click
  const handleEdit = async (id: number) => {
    const faq = await getFAQDetails(id);
    setEditingFAQId(id);
    setValue('question', faq.question);
    setValue('answer', faq.answer);
    setValue('index', faq.index);
    setValue('is_active', faq.is_active);
    setIsModalOpen(true);
  };

  // Handle delete button click
  const handleDelete = () => {
    if (selectedFAQs.length > 0) {
      deleteMutation.mutate(selectedFAQs);
    }
  };

  // Handle checkbox selection
  const handleSelectFAQ = (id: number) => {
    setSelectedFAQs((prev) => (prev.includes(id) ? prev.filter((faqId) => faqId !== id) : [...prev, id]));
  };

  // Toggle FAQ accordion
  const toggleFAQ = (index: number) => {
    setOpenIndices((prev) => {
      const newSet = new Set(prev);
      if (newSet.has(index)) {
        newSet.delete(index);
      } else {
        newSet.add(index);
      }
      return newSet;
    });
  };

  // Handle modal close
  const handleCloseModal = () => {
    setIsModalOpen(false);
    setEditingFAQId(null); // Reset editingFAQId when closing the modal
    reset(); // Reset the form to default values
  };

  return (
    <div className="p-3 min-h-screen" dir="rtl">
      <div className="flex justify-between items-center mb-6">
        <h2 className="text-xl sm:text-2xl font-bold text-doctor-dark font-heading text-start">إدارة الأسئلة الشائعة</h2>
        {/* Actions */}

        <div className="flex justify-center gap-2">
          <button
            onClick={() => {
              setEditingFAQId(null); // Ensure we're in create mode
              reset(); // Reset form to default values
              setIsModalOpen(true);
            }}
            className="flex items-center justify-center gap-2 p-3 bg-green-600 text-white rounded-full hover:bg-green-700 transition-colors duration-200 w-full sm:w-auto cursor-pointer"
          >
            <FaPlus />
          </button>
          <button
            onClick={handleDelete}
            disabled={selectedFAQs.length === 0 || deleteMutation.isPending}
            className="flex items-center justify-center gap-2 p-3 bg-red-600 text-white rounded-full hover:bg-red-700 disabled:opacity-50 transition-colors duration-200 w-full sm:w-auto cursor-pointer"
          >
            {deleteMutation.isPending ? <FaSpinner className="animate-spin" /> : <FaTrash />}
          </button>
        </div>
      </div>

      {/* Error Message */}
      {error && <p className="text-red-500 mb-4 text-center">{handleBackendError(error)}</p>}

      {/* FAQ Accordion */}
      <div>
        {isLoading ? (
          <div className="text-center py-4 text-gray-600">جارٍ التحميل...</div>
        ) : data?.length ? (
          <div className="space-y-4">
            {data.map((faq: FAQ, index: number) => (
              <div key={faq.id} className="border border-gray-200 rounded-lg hover:shadow-sm transition-all duration-300 bg-white">
                <div className="flex items-center justify-between p-4">
                  <div className="flex items-center gap-3">
                    <input
                      type="checkbox"
                      checked={selectedFAQs.includes(faq.id)}
                      onChange={() => handleSelectFAQ(faq.id)}
                      className="h-5 w-5 text-blue-600 rounded focus:ring-blue-500"
                    />
                    <button onClick={() => toggleFAQ(index)} className="flex justify-between items-center w-full text-right focus:outline-none">
                      <h3 className="text-base sm:text-lg font-semibold text-doctor-dark font-heading">
                        {faq.index}. {faq.question}
                      </h3>
                    </button>
                  </div>
                  <div className="flex items-center gap-2">
                    <span className={`inline-block px-2 py-1 text-xs font-semibold rounded-full ${faq.is_active ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>
                      {faq.is_active ? 'نشط' : 'غير نشط'}
                    </span>
                    <button onClick={() => handleEdit(faq.id)} className="text-doctor-dark hover:text-doctor-primary transition-colors duration-200" aria-label="تعديل السؤال">
                      <FaEdit className="text-lg" />
                    </button>
                    <button onClick={() => toggleFAQ(index)} className="text-gray-600 hover:text-gray-800">
                      {openIndices.has(index) ? <FaChevronUp /> : <FaChevronDown />}
                    </button>
                  </div>
                </div>
                <div
                  className={`overflow-hidden transition-all duration-300 ease-in-out ${
                    openIndices.has(index) ? 'max-h-96 opacity-100 translate-y-0' : 'max-h-0 opacity-0 -translate-y-2'
                  }`}
                >
                  <div className="p-4 pt-0 text-gray-600 text-sm sm:text-base font-body space-y-2">
                    <p>{faq.answer}</p>
                  </div>
                </div>
              </div>
            ))}
          </div>
        ) : (
          <div className="text-center py-4 text-gray-600">لا توجد أسئلة شائعة</div>
        )}
      </div>

      {/* Modal for Create/Update */}
      {isModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          {/* Overlay */}
          <div className="absolute inset-0 bg-black opacity-50" onClick={handleCloseModal} />

          {/* Modal */}
          <div className="relative bg-white p-6 rounded-lg w-full max-w-md shadow-lg">
            <h3 className="text-xl font-bold mb-4 text-doctor-dark font-heading">{editingFAQId ? 'تعديل السؤال' : 'إضافة سؤال'}</h3>
            <form onSubmit={handleSubmit(onSubmit)} className="space-y-4" dir="rtl">
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">السؤال</label>
                <input
                  {...register('question')}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-200"
                  placeholder="السؤال"
                />
                {errors.question && <p className="text-red-500 text-sm mt-1">{errors.question.message}</p>}
              </div>

              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الإجابة</label>
                <textarea
                  {...register('answer')}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-200"
                  placeholder="الإجابة"
                  rows={4}
                />
                {errors.answer && <p className="text-red-500 text-sm mt-1">{errors.answer.message}</p>}
              </div>

              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الترتيب</label>
                <input
                  type="number"
                  {...register('index', { valueAsNumber: true })}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-200"
                  placeholder="الترتيب"
                />
                {errors.index && <p className="text-red-500 text-sm mt-1">{errors.index.message}</p>}
              </div>

              {/* Show is_active field only when updating */}
              {editingFAQId && (
                <div>
                  <label className="flex items-center gap-2">
                    <input type="checkbox" {...register('is_active')} className="h-5 w-5 text-blue-600 rounded focus:ring-blue-500" />
                    <span className="text-sm font-medium text-gray-700">نشط</span>
                  </label>
                </div>
              )}

              <div className="flex gap-2">
                <button
                  type="submit"
                  disabled={createMutation.isPending || updateMutation.isPending}
                  className="flex-1 px-4 py-2 bg-sky-600 text-white rounded-lg hover:bg-sky-700 disabled:opacity-50 transition-colors duration-200"
                >
                  {createMutation.isPending || updateMutation.isPending ? 'جارٍ...' : 'حفظ'}
                </button>
                <button type="button" onClick={handleCloseModal} className="flex-1 px-4 py-2 bg-gray-300 rounded-lg hover:bg-gray-400 transition-colors duration-200">
                  إلغاء
                </button>
              </div>

              {(createMutation.isError || updateMutation.isError) && (
                <p className="text-red-500 text-sm mt-2">{handleBackendError(createMutation.error || updateMutation.error)}</p>
              )}
            </form>
          </div>
        </div>
      )}
    </div>
  );
};

export default FAQs;
