import Cookies from 'js-cookie';
import { CreateFAQFormValues, FAQ, UpdateFAQFormValues } from './faq.schema';

const API_URL = process.env.NEXT_PUBLIC_SERVER_URL;

// Helper function to get the auth token from cookies
const getAuthToken = () => {
  if (typeof window !== 'undefined') {
    return Cookies.get('dr_auth_token');
  }
  return null;
};

// List FAQs with filters (sort by index by default)
export const listFAQs = async () => {
  const response = await fetch(`${API_URL}/faq/list`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as FAQ[];
};

// Create a new FAQ
export const createFAQ = async (data: CreateFAQFormValues) => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/faq/create`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as FAQ;
};

// Get FAQ details by ID
export const getFAQDetails = async (id: number) => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/faq/${id}`, {
    method: 'GET',
    headers: {
      Authorization: `Bearer ${token}`,
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as FAQ;
};

// Update an FAQ
export const updateFAQ = async (id: number, data: UpdateFAQFormValues) => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');
  const response = await fetch(`${API_URL}/faq/${id}`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });
  const result = await response.json();
  if (!response.ok) {
    throw result;
  }
  // console.log(result);

  return result as FAQ;
};

// Delete FAQs
export const deleteFAQs = async (ids: number[]) => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/faq/delete-many`, {
    method: 'DELETE',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify({ ids }),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result;
};
