import { z } from 'zod';

// Define the FAQ type based on the backend schema
export interface FAQ {
  id: number;
  question: string;
  answer: string;
  index: number;
  is_active: boolean;
  createdAt: string;
  updatedAt: string;
}

// Base schema for shared fields
const baseFAQSchema = z.object({
  question: z.string().min(1, 'السؤال مطلوب').max(1000, 'السؤال يجب ألا يتجاوز 1000 حرف'),
  answer: z.string().min(1, 'الإجابة مطلوبة').max(1000, 'الإجابة يجب ألا تتجاوز 1000 حرف'),
  index: z.number().int().min(0, 'الترتيب يجب أن يكون رقمًا صحيحًا غير سالب'),
});

// Schema for creating an FAQ (excludes is_active)
export const createFAQFormSchema = baseFAQSchema;

// Schema for updating an FAQ (includes is_active)
export const updateFAQFormSchema = baseFAQSchema.extend({
  is_active: z.boolean().optional(),
});

export type CreateFAQFormValues = z.infer<typeof createFAQFormSchema>;
export type UpdateFAQFormValues = z.infer<typeof updateFAQFormSchema>;

// Define the schema for listing FAQs (filters)
export const listFAQsSchema = z.object({
  page: z.number().min(-1).optional(),
  pageSize: z.number().optional(),
  q: z.string().max(200).optional(),
  order: z
    .array(
      z.object({
        field: z.enum(['id', 'index']),
        orderType: z.enum(['asc', 'desc']),
      }),
    )
    .optional(),
});

export type ListFAQsFilter = z.infer<typeof listFAQsSchema>;

// Define the schema for deleting FAQs
export const deleteFAQsSchema = z.object({
  ids: z.array(z.number().int().positive().min(1)).optional(),
});

export type DeleteFAQsData = z.infer<typeof deleteFAQsSchema>;
