'use client';

import { handleBackendError } from '@/shared/api/errorHandler';
import { zodResolver } from '@hookform/resolvers/zod';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { FaEdit, FaPlus, FaSpinner, FaTrash } from 'react-icons/fa';
import {
  Category,
  CreateCategoryFormValues,
  CreateSubCategoryFormValues,
  UpdateCategoryFormValues,
  createMainCategoryFormSchema,
  createSubCategoryFormSchema,
  updateCategoryFormSchema,
} from './category.schema';
import { createCategory, deleteCategories, getCategoryDetails, listCategories, updateCategory } from './category.service';

const Categories = () => {
  const queryClient = useQueryClient();
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [modalType, setModalType] = useState<'main' | 'sub' | 'edit'>('main');
  const [editingCategoryId, setEditingCategoryId] = useState<number | null>(null);
  const [selectedCategories, setSelectedCategories] = useState<number[]>([]);

  // Form for Main Category
  const mainCategoryForm = useForm<CreateCategoryFormValues>({
    resolver: zodResolver(createMainCategoryFormSchema),
    defaultValues: { name: '', index: 0, is_active: true },
  });

  // Form for Sub-Category
  const subCategoryForm = useForm<CreateSubCategoryFormValues>({
    resolver: zodResolver(createSubCategoryFormSchema),
    defaultValues: { name: '', parent_category_id: undefined, index: 0, is_active: true },
  });

  // Form for Update
  const updateForm = useForm<UpdateCategoryFormValues>({
    resolver: zodResolver(updateCategoryFormSchema),
    defaultValues: { name: '', index: 0, is_active: true },
  });

  // Fetch all categories
  const { data, isLoading, error } = useQuery({
    queryKey: ['categories'],
    queryFn: () => listCategories(),
  });

  // Filter main categories and attach sub-categories
  const mainCategories =
    data
      ?.filter((cat) => !cat.parent_category_id)
      .map((mainCat) => ({
        ...mainCat,
        sub_categories: data.filter((subCat) => subCat.parent_category_id === mainCat.id),
      })) || [];

  // Create category mutation
  const createMutation = useMutation({
    mutationFn: createCategory,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['categories'] });
      setIsModalOpen(false);
      setModalType('main');
      setEditingCategoryId(null);
      mainCategoryForm.reset();
      subCategoryForm.reset();
    },
    onError: (err) => {
      console.error('Create error:', err);
    },
  });

  // Update category mutation
  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: UpdateCategoryFormValues }) => updateCategory(id, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['categories'] });
      setIsModalOpen(false);
      setModalType('main');
      setEditingCategoryId(null);
      updateForm.reset();
    },
    onError: (err) => {
      console.error('Update error:', err);
    },
  });

  // Delete categories mutation
  const deleteMutation = useMutation({
    mutationFn: deleteCategories,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['categories'] });
      setSelectedCategories([]);
    },
    onError: (err) => {
      console.error('Delete error:', err);
    },
  });

  // Handle form submission
  const onSubmit = (data: CreateCategoryFormValues | CreateSubCategoryFormValues | UpdateCategoryFormValues) => {
    if (editingCategoryId) {
      updateMutation.mutate({ id: editingCategoryId, data: data as UpdateCategoryFormValues });
    } else if (modalType === 'main') {
      createMutation.mutate(data as CreateCategoryFormValues);
    } else if (modalType === 'sub') {
      createMutation.mutate(data as CreateSubCategoryFormValues);
    }
  };

  // Handle edit button click
  const handleEdit = async (id: number) => {
    const category = await getCategoryDetails(id);
    setEditingCategoryId(id);
    setModalType('edit');
    updateForm.setValue('name', category.name);
    updateForm.setValue('index', category.index);
    updateForm.setValue('is_active', category.is_active);
    setIsModalOpen(true);
  };

  // Handle delete button click
  const handleDelete = () => {
    if (selectedCategories.length > 0) {
      deleteMutation.mutate(selectedCategories);
    }
  };

  // Handle checkbox selection
  const handleSelectCategory = (id: number) => {
    setSelectedCategories((prev) => (prev.includes(id) ? prev.filter((catId) => catId !== id) : [...prev, id]));
  };

  // Handle modal close
  const handleCloseModal = () => {
    setIsModalOpen(false);
    setModalType('main');
    setEditingCategoryId(null);
    mainCategoryForm.reset();
    subCategoryForm.reset();
    updateForm.reset();
  };

  // Open modal for main category
  const openMainCategoryModal = () => {
    setModalType('main');
    setEditingCategoryId(null);
    mainCategoryForm.reset({ name: '', index: 0, is_active: true });
    setIsModalOpen(true);
  };

  // Open modal for sub-category
  const openSubCategoryModal = () => {
    setModalType('sub');
    setEditingCategoryId(null);
    subCategoryForm.reset({ name: '', parent_category_id: undefined, index: 0, is_active: true });
    setIsModalOpen(true);
  };

  return (
    <div className="px-2 py-4 min-h-screen" dir="rtl">
      {/* Header */}
      <div className="flex flex-col sm:flex-row justify-between items-center mb-6 sm:mb-8 gap-4">
        <h2 className="text-xl sm:text-2xl font-bold text-doctor-dark">إدارة الأقسام الطبية</h2>
        <div className="flex flex-col sm:flex-row gap-2 sm:gap-3 w-full sm:w-auto">
          <button
            onClick={openMainCategoryModal}
            className="flex items-center justify-center gap-2 px-3 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 transition-colors duration-200 w-full sm:w-auto text-sm sm:text-base"
          >
            <FaPlus /> <span>فئة رئيسية</span>
          </button>
          <button
            onClick={openSubCategoryModal}
            className="flex items-center justify-center gap-2 px-3 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors duration-200 w-full sm:w-auto text-sm sm:text-base"
          >
            <FaPlus /> <span>فئة فرعية</span>
          </button>
          <button
            onClick={handleDelete}
            disabled={selectedCategories.length === 0 || deleteMutation.isPending}
            className="flex items-center justify-center gap-2 px-3 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 disabled:opacity-50 transition-colors duration-200 w-full sm:w-auto text-sm sm:text-base"
          >
            {deleteMutation.isPending ? <FaSpinner className="animate-spin" /> : <FaTrash />}
            <span>حذف المحدد</span>
          </button>
        </div>
      </div>

      {/* Error Message */}
      {error && <p className="text-red-500 mb-4 text-center text-sm sm:text-base">{handleBackendError(error)}</p>}

      {/* Categories List */}
      <div>
        {isLoading ? (
          <div className="text-center py-4 text-gray-600 text-sm sm:text-base">جارٍ التحميل...</div>
        ) : mainCategories.length ? (
          <div className="space-y-4 sm:space-y-6">
            {mainCategories.map((category: Category) => (
              <div key={category.id} className="bg-white rounded-lg border border-neutral-200 p-3 sm:p-4 transition-shadow duration-200">
                {/* Main Category */}
                <div className="flex items-center justify-between gap-2">
                  <div className="flex items-center gap-2 sm:gap-3 w-full sm:w-auto">
                    <input
                      type="checkbox"
                      checked={selectedCategories.includes(category.id)}
                      onChange={() => handleSelectCategory(category.id)}
                      className="h-4 w-4 sm:h-5 sm:w-5 text-blue-600 rounded focus:ring-blue-500"
                    />
                    <div className="flex items-center gap-2">
                      <h3 className="text-base sm:text-lg font-semibold text-gray-800 truncate">{category.name}</h3>
                      <span className={`inline-block px-2 py-1 text-xs font-medium rounded-full ${category.is_active ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>
                        {category.is_active ? 'نشط' : 'غير نشط'}
                      </span>
                    </div>
                  </div>
                  <div className="flex items-center gap-2">
                    <button onClick={() => handleEdit(category.id)} className="text-gray-600 hover:text-blue-600 transition-colors duration-200" aria-label="تعديل الفئة">
                      <FaEdit className="text-base sm:text-lg" />
                    </button>
                  </div>
                </div>

                {/* Sub-categories */}
                {category.sub_categories && category.sub_categories.length > 0 && (
                  <div className="mt-3 sm:mt-4 space-y-2">
                    {category.sub_categories.map((subCat: Category) => (
                      <div
                        key={subCat.id}
                        className="flex items-center justify-between bg-gray-100 p-2 sm:p-3 rounded-md hover:bg-gray-200 transition-colors duration-200 ml-0 sm:ml-6"
                      >
                        <div className="flex items-center gap-2 sm:gap-3 w-full sm:w-auto">
                          <input
                            type="checkbox"
                            checked={selectedCategories.includes(subCat.id)}
                            onChange={() => handleSelectCategory(subCat.id)}
                            className="h-4 w-4 sm:h-5 sm:w-5 text-blue-600 rounded focus:ring-blue-500"
                          />
                          <div className="flex items-center gap-2">
                            <span className="text-sm font-medium text-gray-700 truncate">{subCat.name}</span>
                            <span
                              className={`inline-block px-2 py-1 text-xs font-medium rounded-full ${subCat.is_active ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}
                            >
                              {subCat.is_active ? 'نشط' : 'غير نشط'}
                            </span>
                          </div>
                        </div>
                        <button onClick={() => handleEdit(subCat.id)} className="text-gray-600 hover:text-blue-600 transition-colors duration-200">
                          <FaEdit className="text-base sm:text-lg" />
                        </button>
                      </div>
                    ))}
                  </div>
                )}
              </div>
            ))}
          </div>
        ) : (
          <div className="text-center py-4 text-gray-600 text-sm sm:text-base">لا توجد فئات</div>
        )}
      </div>

      {/* Modal for Create/Update */}
      {isModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50 px-4 sm:px-0">
          {/* Overlay */}
          <div className="absolute inset-0 bg-black opacity-50" onClick={handleCloseModal} />

          {/* Modal */}
          <div className="relative bg-white p-4 sm:p-6 rounded-lg w-full max-w-md shadow-lg">
            <h3 className="text-lg sm:text-xl font-bold mb-4 text-gray-800">
              {modalType === 'edit' ? 'تعديل الفئة' : modalType === 'main' ? 'إضافة فئة رئيسية' : 'إضافة فئة فرعية'}
            </h3>

            {/* Main Category Form */}
            {modalType === 'main' && (
              <form onSubmit={mainCategoryForm.handleSubmit(onSubmit)} className="space-y-4" dir="rtl">
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">اسم الفئة</label>
                  <input
                    {...mainCategoryForm.register('name')}
                    className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-200 text-sm sm:text-base"
                    placeholder="اسم الفئة"
                  />
                  {mainCategoryForm.formState.errors.name && <p className="text-red-500 text-xs sm:text-sm mt-1">{mainCategoryForm.formState.errors.name.message}</p>}
                </div>
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">الترتيب</label>
                  <input
                    type="number"
                    {...mainCategoryForm.register('index', { valueAsNumber: true })}
                    className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-200 text-sm sm:text-base"
                    placeholder="0"
                  />
                  {mainCategoryForm.formState.errors.index && <p className="text-red-500 text-xs sm:text-sm mt-1">{mainCategoryForm.formState.errors.index.message}</p>}
                </div>
                <div>
                  <label className="flex items-center gap-2 text-sm font-medium text-gray-700">
                    <input type="checkbox" {...mainCategoryForm.register('is_active')} className="h-4 w-4 text-blue-600 rounded focus:ring-blue-500" />
                    نشط
                  </label>
                </div>
                <div className="flex flex-col sm:flex-row gap-2">
                  <button
                    type="submit"
                    disabled={createMutation.isPending}
                    className="flex-1 px-3 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors duration-200 text-sm sm:text-base"
                  >
                    {createMutation.isPending ? 'جارٍ...' : 'حفظ'}
                  </button>
                  <button
                    type="button"
                    onClick={handleCloseModal}
                    className="flex-1 px-3 py-2 bg-gray-300 rounded-lg hover:bg-gray-400 transition-colors duration-200 text-sm sm:text-base"
                  >
                    إلغاء
                  </button>
                </div>
                {createMutation.isError && <p className="text-red-500 text-xs sm:text-sm mt-2">{handleBackendError(createMutation.error)}</p>}
              </form>
            )}

            {/* Sub-Category Form */}
            {modalType === 'sub' && (
              <form onSubmit={subCategoryForm.handleSubmit(onSubmit)} className="space-y-4" dir="rtl">
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">اسم الفئة</label>
                  <input
                    {...subCategoryForm.register('name')}
                    className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-200 text-sm sm:text-base"
                    placeholder="اسم الفئة"
                  />
                  {subCategoryForm.formState.errors.name && <p className="text-red-500 text-xs sm:text-sm mt-1">{subCategoryForm.formState.errors.name.message}</p>}
                </div>
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">الفئة الأصلية</label>
                  <select
                    {...subCategoryForm.register('parent_category_id', { valueAsNumber: true })}
                    className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-200 text-sm sm:text-base"
                  >
                    <option value="">اختر الفئة الأصلية</option>
                    {mainCategories
                      .filter((cat) => cat.is_active)
                      .map((cat) => (
                        <option key={cat.id} value={cat.id}>
                          {cat.name}
                        </option>
                      ))}
                  </select>
                  {subCategoryForm.formState.errors.parent_category_id && (
                    <p className="text-red-500 text-xs sm:text-sm mt-1">{subCategoryForm.formState.errors.parent_category_id.message}</p>
                  )}
                </div>
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">الترتيب</label>
                  <input
                    type="number"
                    {...subCategoryForm.register('index', { valueAsNumber: true })}
                    className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-200 text-sm sm:text-base"
                    placeholder="0"
                  />
                  {subCategoryForm.formState.errors.index && <p className="text-red-500 text-xs sm:text-sm mt-1">{subCategoryForm.formState.errors.index.message}</p>}
                </div>
                <div>
                  <label className="flex items-center gap-2 text-sm font-medium text-gray-700">
                    <input type="checkbox" {...subCategoryForm.register('is_active')} className="h-4 w-4 text-blue-600 rounded focus:ring-blue-500" />
                    نشط
                  </label>
                </div>
                <div className="flex flex-col sm:flex-row gap-2">
                  <button
                    type="submit"
                    disabled={createMutation.isPending}
                    className="flex-1 px-3 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors duration-200 text-sm sm:text-base"
                  >
                    {createMutation.isPending ? 'جارٍ...' : 'حفظ'}
                  </button>
                  <button
                    type="button"
                    onClick={handleCloseModal}
                    className="flex-1 px-3 py-2 bg-gray-300 rounded-lg hover:bg-gray-400 transition-colors duration-200 text-sm sm:text-base"
                  >
                    إلغاء
                  </button>
                </div>
                {createMutation.isError && <p className="text-red-500 text-xs sm:text-sm mt-2">{handleBackendError(createMutation.error)}</p>}
              </form>
            )}

            {/* Update Form */}
            {modalType === 'edit' && (
              <form onSubmit={updateForm.handleSubmit(onSubmit)} className="space-y-4" dir="rtl">
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">اسم الفئة</label>
                  <input
                    {...updateForm.register('name')}
                    className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-200 text-sm sm:text-base"
                    placeholder="اسم الفئة"
                  />
                  {updateForm.formState.errors.name && <p className="text-red-500 text-xs sm:text-sm mt-1">{updateForm.formState.errors.name.message}</p>}
                </div>
                <div>
                  <label className="block mb-1 text-sm font-medium text-gray-700">الترتيب</label>
                  <input
                    type="number"
                    {...updateForm.register('index', { valueAsNumber: true })}
                    className="w-full px-3 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-blue-500 transition-all duration-200 text-sm sm:text-base"
                    placeholder="0"
                  />
                  {updateForm.formState.errors.index && <p className="text-red-500 text-xs sm:text-sm mt-1">{updateForm.formState.errors.index.message}</p>}
                </div>
                <div>
                  <label className="flex items-center gap-2 text-sm font-medium text-gray-700">
                    <input type="checkbox" {...updateForm.register('is_active')} className="h-4 w-4 text-blue-600 rounded focus:ring-blue-500" />
                    نشط
                  </label>
                </div>
                <div className="flex flex-col sm:flex-row gap-2">
                  <button
                    type="submit"
                    disabled={updateMutation.isPending}
                    className="flex-1 px-3 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors duration-200 text-sm sm:text-base"
                  >
                    {updateMutation.isPending ? 'جارٍ...' : 'حفظ'}
                  </button>
                  <button
                    type="button"
                    onClick={handleCloseModal}
                    className="flex-1 px-3 py-2 bg-gray-300 rounded-lg hover:bg-gray-400 transition-colors duration-200 text-sm sm:text-base"
                  >
                    إلغاء
                  </button>
                </div>
                {updateMutation.isError && <p className="text-red-500 text-xs sm:text-sm mt-2">{handleBackendError(updateMutation.error)}</p>}
              </form>
            )}
          </div>
        </div>
      )}
    </div>
  );
};

export default Categories;
