import Cookies from 'js-cookie';
import { CreateCategoryFormValues, Category, UpdateCategoryFormValues } from './category.schema';

const API_URL = process.env.NEXT_PUBLIC_SERVER_URL || 'http://localhost:3000';

// Helper function to get the auth token from cookies
const getAuthToken = () => {
  if (typeof window !== 'undefined') {
    return Cookies.get('dr_auth_token');
  }
  return null;
};

// List all categories with optional type and parent_id filters
export const listCategories = async (type?: string, parentId?: number) => {
  const url = new URL(`${API_URL}/category/list`);
  if (type) url.searchParams.append('type', type);
  if (parentId) url.searchParams.append('parent_id', parentId.toString());

  const response = await fetch(url.toString(), {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Category[];
};

// Create a new category
export const createCategory = async (data: CreateCategoryFormValues) => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/category/create`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Category;
};

// Get category details by ID
export const getCategoryDetails = async (id: number) => {
  const response = await fetch(`${API_URL}/category/${id}`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Category;
};

// Update a category
export const updateCategory = async (id: number, data: UpdateCategoryFormValues) => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/category/${id}`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Category;
};

// Delete categories
export const deleteCategories = async (ids: number[]) => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/category/delete-many`, {
    method: 'DELETE',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify({ ids }),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result;
};
