/* eslint-disable @typescript-eslint/no-explicit-any */
'use client';

import { zodResolver } from '@hookform/resolvers/zod';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { FaEdit, FaImage, FaPlus, FaTrash } from 'react-icons/fa';
import { createCarouselFormSchema, CreateCarouselFormValues, updateCarouselFormSchema, UpdateCarouselFormValues } from './carousel.schema';
import { createCarousel, deleteCarousel, getCarouselDetails, listCarousel, updateCarousel, uploadFile, removeFile } from './carousel.service';
import Image from 'next/image';

const Carousel = () => {
  const queryClient = useQueryClient();
  const [selectedIds, setSelectedIds] = useState<number[]>([]);
  const [isCreateModalOpen, setIsCreateModalOpen] = useState(false);
  const [isEditModalOpen, setIsEditModalOpen] = useState(false);
  const [editingCarouselId, setEditingCarouselId] = useState<number | null>(null);
  const [imgPreview, setImgPreview] = useState<string | null>(null);
  const [uploadError, setUploadError] = useState<string | null>(null);
  const [isImgLoading, setIsImgLoading] = useState(false); // Loading state for image

  // Fetch all carousel items
  const {
    data: carouselItems,
    isLoading,
    error,
  } = useQuery({
    queryKey: ['carousel'],
    queryFn: listCarousel,
  });

  // Form setup for create
  const createForm = useForm<CreateCarouselFormValues>({
    resolver: zodResolver(createCarouselFormSchema),
    defaultValues: {
      title: '',
      description: '',
      img: [],
      button_txt: '',
      button_url: '',
    },
  });

  // Form setup for update
  const updateForm = useForm<UpdateCarouselFormValues>({
    resolver: zodResolver(updateCarouselFormSchema),
    defaultValues: {
      title: '',
      description: '',
      img: [],
      button_txt: '',
      button_url: '',
    },
  });

  // Create mutation
  const createMutation = useMutation({
    mutationFn: createCarousel,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['carousel'] });
      setIsCreateModalOpen(false);
      setImgPreview(null);
      setUploadError(null);
      setIsImgLoading(false);
      createForm.reset();
    },
    onError: (err: any) => {
      console.error('Create carousel error:', err);
      setUploadError('فشل في إنشاء عنصر الكاروسيل: ' + (err.message || 'خطأ غير معروف'));
    },
  });

  // Update mutation
  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: UpdateCarouselFormValues }) => updateCarousel(id, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['carousel'] });
      setIsEditModalOpen(false);
      setEditingCarouselId(null);
      setImgPreview(null);
      setUploadError(null);
      setIsImgLoading(false);
      updateForm.reset();
    },
    onError: (err: any) => {
      console.error('Update carousel error:', err);
      setUploadError('فشل في تحديث عنصر الكاروسيل: ' + (err.message || 'خطأ غير معروف'));
    },
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: deleteCarousel,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['carousel'] });
      setSelectedIds([]);
    },
    onError: (err: any) => {
      console.error('Delete carousel error:', err);
      setUploadError('فشل في حذف عناصر الكاروسيل: ' + (err.message || 'خطأ غير معروف'));
    },
  });

  const handleSelectCarousel = (id: number) => {
    setSelectedIds((prev) => (prev.includes(id) ? prev.filter((item) => item !== id) : [...prev, id]));
  };

  const handleCreateSubmit = (data: CreateCarouselFormValues) => {
    createMutation.mutate(data);
  };

  const handleEditClick = async (id: number) => {
    try {
      const carousel = await getCarouselDetails(id);
      setEditingCarouselId(id);
      updateForm.reset({
        title: carousel.title,
        description: carousel.description,
        img: carousel.img,
        button_txt: carousel.button_txt || '',
        button_url: carousel.button_url || '',
      });
      setImgPreview(carousel.img[0]?.url || null);
      setIsEditModalOpen(true);
      setUploadError(null);
      setIsImgLoading(false);
    } catch (err) {
      console.error('Error fetching carousel details:', err);
      setUploadError('فشل في جلب تفاصيل عنصر الكاروسيل');
    }
  };

  const handleUpdateSubmit = (data: UpdateCarouselFormValues) => {
    if (editingCarouselId) {
      updateMutation.mutate({ id: editingCarouselId, data });
    }
  };

  const handleDeleteSelected = () => {
    if (selectedIds.length > 0) {
      deleteMutation.mutate(selectedIds);
    }
  };

  const handleImgChange = async (e: React.ChangeEvent<HTMLInputElement>, form: any) => {
    const file = e.target.files?.[0];
    if (file) {
      try {
        setIsImgLoading(true); // Show spinner
        setUploadError(null);

        // Attempt to remove the old image (if it exists), but don't block upload
        const existingImg = form.getValues('img');
        if (existingImg && existingImg.length > 0) {
          const oldFileUrl = existingImg[0].url;
          removeFile(oldFileUrl);
          // .then(() => console.log(`Successfully removed old image: ${oldFileUrl}`))
          // .catch((err) => console.error(`Failed to remove old image: ${oldFileUrl}`, err));
          // Note: We don’t await here, so removal runs in the background
        }

        // Upload the new image regardless of removal success/failure
        const uploadResponse = await uploadFile(file);
        const imgData = uploadResponse.data.files;
        if (imgData.length === 0) {
          throw new Error('No image data returned');
        }
        form.setValue('img', imgData);
        setImgPreview(imgData[0].url);
      } catch (err: any) {
        console.error('Image upload error:', err);
        setUploadError('فشل في رفع الصورة: ' + (err.message || 'خطأ غير معروف'));
        form.setValue('img', []);
      } finally {
        setIsImgLoading(false); // Hide spinner
      }
    } else {
      setUploadError('يرجى اختيار صورة');
      form.setValue('img', []);
      setIsImgLoading(false);
    }
  };

  const openCreateModal = () => {
    createForm.reset();
    setImgPreview(null);
    setUploadError(null);
    setIsImgLoading(false);
    setIsCreateModalOpen(true);
  };

  return (
    <div className="px-2 py-4 min-h-screen" dir="rtl">
      <h2 className="text-xl sm:text-2xl font-bold mb-6 text-doctor-dark font-heading text-start">إدارة الصور الرئيسية</h2>

      {/* Actions */}
      <div className="flex flex-row-reverse gap-4 mb-6">
        {selectedIds.length > 0 && (
          <button
            onClick={handleDeleteSelected}
            className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 flex items-center gap-2"
            disabled={deleteMutation.isPending}
          >
            <FaTrash /> حذف ({selectedIds.length})
          </button>
        )}
        <button onClick={openCreateModal} className="px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary flex items-center gap-2">
          <FaPlus /> إضافة
        </button>
      </div>

      {/* Carousel List */}
      {isLoading ? (
        <div className="text-center py-8 text-gray-600 text-lg">جارٍ التحميل...</div>
      ) : error ? (
        <div className="text-center py-8 text-red-500 text-lg">حدث خطأ أثناء جلب البيانات: {JSON.stringify(error)}</div>
      ) : carouselItems && carouselItems.length > 0 ? (
        <div className="space-y-6">
          {carouselItems.map((carousel) => (
            <div
              key={carousel.id}
              className={`flex flex-col md:flex-row items-center gap-6 p-6 bg-white border border-neutral-200 rounded-lg hover:shadow-sm transition-all duration-300 ${
                selectedIds.includes(carousel.id) ? 'border-doctor-dark border-2' : ''
              }`}
            >
              {/* Image */}
              {carousel.img && carousel.img.length > 0 ? (
                <div className="relative w-full md:w-40 h-40 rounded-lg overflow-hidden">
                  <Image width={1000} height={1000} src={carousel.img[0].url} alt={carousel.title} className="w-full h-full object-cover" />
                </div>
              ) : (
                <div className="w-full md:w-40 h-40 bg-gray-200 rounded-lg flex items-center justify-center">
                  <span className="text-gray-500">بدون صورة</span>
                </div>
              )}

              {/* Text Content */}
              <div className="flex-1">
                <div className="flex items-center justify-between">
                  <h3 className="text-xl font-semibold text-doctor-dark font-heading mb-2">{carousel.title}</h3>
                  <div className="flex gap-3">
                    <button onClick={() => handleEditClick(carousel.id)} className="text-doctor-dark hover:text-doctor-primary">
                      <FaEdit size={22} />
                    </button>
                    <input
                      type="checkbox"
                      className="h-5 w-5 text-blue-600 rounded focus:ring-blue-500"
                      checked={selectedIds.includes(carousel.id)}
                      onChange={() => handleSelectCarousel(carousel.id)}
                    />
                  </div>
                </div>
                <p className="text-gray-600 text-base leading-relaxed mb-2">{carousel.description}</p>
                {carousel.button_txt && carousel.button_url && (
                  <a
                    href={carousel.button_url}
                    target="_blank"
                    rel="noopener noreferrer"
                    className="inline-block px-3 py-1 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary"
                  >
                    {carousel.button_txt}
                  </a>
                )}
              </div>
            </div>
          ))}
        </div>
      ) : (
        <div className="text-center py-8 text-gray-600 text-lg">لا توجد عناصر كاروسيل</div>
      )}

      {/* Create Modal */}
      {isCreateModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsCreateModalOpen(false)} />
          <div className="relative bg-white p-6 sm:p-8 rounded-xl w-full max-w-md shadow-2xl">
            <h3 className="text-xl font-bold mb-4 text-doctor-dark">إضافة عنصر كاروسيل جديد</h3>
            {uploadError && <p className="text-red-500 text-sm mb-4">{uploadError}</p>}
            <form onSubmit={createForm.handleSubmit(handleCreateSubmit)} className="space-y-4" dir="rtl">
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">العنوان</label>
                <input {...createForm.register('title')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {createForm.formState.errors.title && <p className="text-red-500 text-sm mt-1">{createForm.formState.errors.title.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الوصف</label>
                <textarea
                  {...createForm.register('description')}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark"
                />
                {createForm.formState.errors.description && <p className="text-red-500 text-sm mt-1">{createForm.formState.errors.description.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الصورة</label>
                <label className="w-full px-4 py-2 border rounded-lg flex items-center gap-2 cursor-pointer">
                  <FaImage /> اختر صورة
                  <input type="file" accept="image/*" className="hidden" onChange={(e) => handleImgChange(e, createForm)} />
                </label>
                {isImgLoading ? (
                  <div className="mt-2 w-32 h-32 flex items-center justify-center">
                    <div className="animate-spin rounded-full h-8 w-8 border-t-2 border-b-2 border-doctor-primary"></div>
                  </div>
                ) : imgPreview ? (
                  <Image width={1000} height={1000} src={imgPreview} alt="Preview" className="w-32 h-32 object-cover rounded-lg mt-2" />
                ) : null}
                {createForm.formState.errors.img && <p className="text-red-500 text-sm mt-1">{createForm.formState.errors.img.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">نص الزر</label>
                <input {...createForm.register('button_txt')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {createForm.formState.errors.button_txt && <p className="text-red-500 text-sm mt-1">{createForm.formState.errors.button_txt.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">رابط الزر</label>
                <input {...createForm.register('button_url')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {createForm.formState.errors.button_url && <p className="text-red-500 text-sm mt-1">{createForm.formState.errors.button_url.message}</p>}
              </div>
              <div className="flex gap-3">
                <button
                  type="submit"
                  disabled={createMutation.isPending || isImgLoading}
                  className="flex-1 px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary disabled:opacity-50"
                >
                  {createMutation.isPending ? 'جارٍ...' : 'إضافة'}
                </button>
                <button type="button" onClick={() => setIsCreateModalOpen(false)} className="flex-1 px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400">
                  إلغاء
                </button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* Edit Modal */}
      {isEditModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsEditModalOpen(false)} />
          <div className="relative bg-white p-6 sm:p-8 rounded-xl w-full max-w-md shadow-2xl">
            <h3 className="text-xl font-bold mb-4 text-doctor-dark">تعديل عنصر الكاروسيل</h3>
            {uploadError && <p className="text-red-500 text-sm mb-4">{uploadError}</p>}
            <form onSubmit={updateForm.handleSubmit(handleUpdateSubmit)} className="space-y-4" dir="rtl">
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">العنوان</label>
                <input {...updateForm.register('title')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {updateForm.formState.errors.title && <p className="text-red-500 text-sm mt-1">{updateForm.formState.errors.title.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الوصف</label>
                <textarea
                  {...updateForm.register('description')}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark"
                />
                {updateForm.formState.errors.description && <p className="text-red-500 text-sm mt-1">{updateForm.formState.errors.description.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الصورة</label>
                <label className="w-full px-4 py-2 border rounded-lg flex items-center gap-2 cursor-pointer">
                  <FaImage /> اختر صورة
                  <input type="file" accept="image/*" className="hidden" onChange={(e) => handleImgChange(e, updateForm)} />
                </label>
                {isImgLoading ? (
                  <div className="mt-2 w-32 h-32 flex items-center justify-center">
                    <div className="animate-spin rounded-full h-8 w-8 border-t-2 border-b-2 border-doctor-primary"></div>
                  </div>
                ) : imgPreview ? (
                  <Image width={1000} height={1000} src={imgPreview} alt="Preview" className="w-32 h-32 object-cover rounded-lg mt-2" />
                ) : null}
                {updateForm.formState.errors.img && <p className="text-red-500 text-sm mt-1">{updateForm.formState.errors.img.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">نص الزر</label>
                <input {...updateForm.register('button_txt')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {updateForm.formState.errors.button_txt && <p className="text-red-500 text-sm mt-1">{updateForm.formState.errors.button_txt.message}</p>}
              </div>
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">رابط الزر</label>
                <input {...updateForm.register('button_url')} className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark" />
                {updateForm.formState.errors.button_url && <p className="text-red-500 text-sm mt-1">{updateForm.formState.errors.button_url.message}</p>}
              </div>
              <div className="flex gap-3">
                <button
                  type="submit"
                  disabled={updateMutation.isPending || isImgLoading}
                  className="flex-1 px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary disabled:opacity-50"
                >
                  {updateMutation.isPending ? 'جارٍ...' : 'حفظ'}
                </button>
                <button type="button" onClick={() => setIsEditModalOpen(false)} className="flex-1 px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400">
                  إلغاء
                </button>
              </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
};

export default Carousel;
