import Cookies from 'js-cookie';
import { CreateCarouselFormValues, UpdateCarouselFormValues, Carousel, FileMetadata } from './carousel.schema';

const API_URL = process.env.NEXT_PUBLIC_SERVER_URL || 'http://localhost:3000';

// Helper function to get the auth token from cookies
const getAuthToken = () => {
  if (typeof window !== 'undefined') {
    return Cookies.get('dr_auth_token');
  }
  return null;
};

// List all carousel items (no pagination or filters)
export const listCarousel = async (): Promise<Carousel[]> => {
  const response = await fetch(`${API_URL}/carousel/list`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Carousel[];
};

// Create a new carousel item
export const createCarousel = async (data: CreateCarouselFormValues): Promise<Carousel> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/carousel/create`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Carousel;
};

// Get carousel item details by ID
export const getCarouselDetails = async (id: number): Promise<Carousel> => {
  const response = await fetch(`${API_URL}/carousel/${id}`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Carousel;
};

// Update carousel item
export const updateCarousel = async (id: number, data: UpdateCarouselFormValues): Promise<Carousel> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/carousel/${id}`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Carousel;
};

// Delete multiple carousel items
export const deleteCarousel = async (ids: number[]): Promise<void> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/carousel/delete-many`, {
    method: 'DELETE',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify({ ids }),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result;
};

// Upload file (image)
export const uploadFile = async (file: File): Promise<{ message: string; data: { files: FileMetadata[] } }> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const formData = new FormData();
  formData.append('files', file);

  const response = await fetch(`${API_URL}/multerUploader/upload`, {
    method: 'POST',
    headers: {
      Authorization: `Bearer ${token}`,
    },
    body: formData,
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as { message: string; data: { files: FileMetadata[] } };
};

// Remove file (delete old image from server)
export const removeFile = async (fileUrl: string): Promise<void> => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  await fetch(`${API_URL}/multerUploader/remove`, {
    method: 'DELETE',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify({
      files: [{ url: fileUrl }],
    }),
  });
};
