'use client';

import { handleBackendError } from '@/shared/api/errorHandler';
import { zodResolver } from '@hookform/resolvers/zod';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import Link from 'next/link';
import { useState } from 'react';
import { useForm } from 'react-hook-form';
import * as FaIcons from 'react-icons/fa';
import { FaEdit } from 'react-icons/fa';
import { Account, updateAccountFormSchema, UpdateAccountFormValues } from './accounts.schema';
import { getAccountDetails, listAccounts, updateAccount } from './accounts.service';

// Type for the FaIcons object
// eslint-disable-next-line @typescript-eslint/no-explicit-any
type FaIconType = { [key: string]: React.ComponentType<any> };

const Accounts = () => {
  const queryClient = useQueryClient();
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingAccountId, setEditingAccountId] = useState<number | null>(null);

  // Form setup (only for updating)
  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
    setValue,
  } = useForm<UpdateAccountFormValues>({
    resolver: zodResolver(updateAccountFormSchema),
    defaultValues: {
      url: '',
      is_active: true,
    },
  });

  // Fetch accounts
  const { data, isLoading, error } = useQuery({
    queryKey: ['accounts'],
    queryFn: listAccounts,
  });

  // Update account mutation
  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: UpdateAccountFormValues }) => updateAccount(id, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['accounts'] });
      setIsModalOpen(false);
      setEditingAccountId(null);
      reset();
    },
    onError: (err) => {
      console.error('Update error:', err);
    },
  });

  // Handle form submission
  const onSubmit = (data: UpdateAccountFormValues) => {
    if (editingAccountId) {
      updateMutation.mutate({ id: editingAccountId, data });
    }
  };

  // Handle edit button click
  const handleEdit = async (id: number) => {
    const account = await getAccountDetails(id);
    setEditingAccountId(id);
    setValue('url', account.url);
    setValue('is_active', account.is_active);
    setIsModalOpen(true);
  };

  // Function to dynamically render the icon inside a circle
  const renderIcon = (iconName: string) => {
    const IconComponent = (FaIcons as FaIconType)[iconName];
    return IconComponent ? (
      <div className="flex items-center justify-center w-12 h-12 sm:w-14 sm:h-14 rounded-full bg-doctor-dark border border-gray-300 shadow-sm">
        <IconComponent className="text-2xl sm:text-3xl text-gray-50" />
      </div>
    ) : (
      <div className="flex items-center justify-center w-12 h-12 sm:w-14 sm:h-14 rounded-full bg-gray-100 border border-gray-300 shadow-sm">
        <span className="text-2xl sm:text-3xl text-gray-500">-</span>
      </div>
    );
  };

  return (
    <div className="px-2 py-4 min-h-screen" dir="rtl">
      <h2 className="text-xl sm:text-2xl font-bold mb-6 text-doctor-dark font-heading text-start">إدارة السوشيال ميديا</h2>

      {/* Error Message */}
      {error && <p className="text-red-500 mb-6 text-center text-sm sm:text-base">{handleBackendError(error)}</p>}

      {/* Accounts List */}
      {isLoading ? (
        <div className="text-center py-8 text-gray-600 text-lg">جارٍ التحميل...</div>
      ) : data?.length ? (
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
          {data.map((account: Account) => (
            <div
              key={account.id}
              className="bg-white rounded-xl p-4 sm:p-6 flex flex-col items-center justify-between transition duration-300 border border-neutral-200 hover:shadow-md"
            >
              {/* Circular Icon */}
              <div className="mb-4">{renderIcon(account.icon)}</div>

              {/* URL */}
              <Link
                href={account.url}
                target="_blank"
                rel="noopener noreferrer"
                className="text-doctor-dark hover:text-doctor-primary text-sm sm:text-base font-medium truncate w-full text-center hover:underline"
              >
                {account.url}
              </Link>

              {/* Status */}
              <div className="mt-4 flex items-center gap-2">
                <span className={`inline-block px-2 py-1 text-xs font-semibold rounded-full ${account.is_active ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>
                  {account.is_active ? 'نشط' : 'غير نشط'}
                </span>
              </div>

              {/* Edit Button */}
              <button
                onClick={() => handleEdit(account.id)}
                className="mt-4 px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary transition-colors duration-200 flex items-center gap-2 text-sm sm:text-base"
              >
                <FaEdit /> تعديل
              </button>
            </div>
          ))}
        </div>
      ) : (
        <div className="text-center py-8 text-gray-600 text-lg">لا توجد حسابات</div>
      )}

      {/* Modal for Update */}
      {isModalOpen && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          {/* Overlay */}
          <div className="absolute inset-0 bg-black opacity-50" onClick={() => setIsModalOpen(false)} />

          {/* Modal */}
          <div className="relative bg-white p-6 sm:p-8 rounded-xl w-full max-w-md shadow-2xl transform transition-all duration-300 scale-100">
            <h3 className="text-xl sm:text-2xl font-bold mb-6 text-doctor-dark font-heading">تعديل الحساب</h3>
            <form onSubmit={handleSubmit(onSubmit)} className="space-y-6" dir="rtl">
              <div>
                <label className="block mb-1 text-sm font-medium text-gray-700">الرابط</label>
                <input
                  {...register('url')}
                  className="w-full px-4 py-2 border rounded-lg text-right focus:outline-none focus:ring-2 focus:ring-doctor-dark transition-all duration-200 text-sm sm:text-base"
                  placeholder="الرابط"
                />
                {errors.url && <p className="text-red-500 text-sm mt-1">{errors.url.message}</p>}
              </div>

              <div>
                <label className="flex items-center gap-2 text-sm sm:text-base text-gray-700">
                  <input type="checkbox" {...register('is_active')} className="w-4 h-4 text-doctor-dark focus:ring-doctor-dark border-gray-300 rounded" />
                  نشط
                </label>
              </div>

              <div className="flex gap-3">
                <button
                  type="submit"
                  disabled={updateMutation.isPending}
                  className="flex-1 px-4 py-2 bg-doctor-dark text-white rounded-lg hover:bg-doctor-primary disabled:opacity-50 transition-colors duration-200 text-sm sm:text-base"
                >
                  {updateMutation.isPending ? 'جارٍ...' : 'حفظ'}
                </button>
                <button
                  type="button"
                  onClick={() => {
                    setIsModalOpen(false);
                    setEditingAccountId(null);
                    reset();
                  }}
                  className="flex-1 px-4 py-2 bg-gray-300 text-gray-700 rounded-lg hover:bg-gray-400 transition-colors duration-200 text-sm sm:text-base"
                >
                  إلغاء
                </button>
              </div>

              {updateMutation.isError && <p className="text-red-500 text-sm mt-2">{handleBackendError(updateMutation.error)}</p>}
            </form>
          </div>
        </div>
      )}
    </div>
  );
};

export default Accounts;
