import { Account, UpdateAccountFormValues } from './accounts.schema';
import Cookies from 'js-cookie';

const API_URL = process.env.NEXT_PUBLIC_SERVER_URL;

// Helper function to get the auth token from cookies
const getAuthToken = () => {
  if (typeof window !== 'undefined') {
    return Cookies.get('dr_auth_token');
  }
  return null;
};

// List all accounts (no filters, pagination, or searching)
export const listAccounts = async () => {
  const response = await fetch(`${API_URL}/account/list`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  const data = result.data;
  return data as Account[];
};

// Get account details by ID
export const getAccountDetails = async (id: number) => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/account/${id}`, {
    method: 'GET',
    headers: {
      Authorization: `Bearer ${token}`,
    },
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Account;
};

// Update an account
export const updateAccount = async (id: number, data: UpdateAccountFormValues) => {
  const token = getAuthToken();
  if (!token) throw new Error('No auth token found');

  const response = await fetch(`${API_URL}/account/${id}`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      Authorization: `Bearer ${token}`,
    },
    body: JSON.stringify(data),
  });

  const result = await response.json();
  if (!response.ok) {
    throw result;
  }

  return result as Account;
};
